<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace AlloCine\OrmsBundle\Logger;

use Psr\Log\LoggerInterface as SfLoggerInterface;
use Orms\Logger\LoggerInterface;

/**
 * DbalLogger.
 */
class OrmsLogger implements LoggerInterface
{
    protected $logger;

    /**
     * @var string
     */
    protected $lastExecutedQuery;

    /** @var array $queries Executed SQL queries. */
    public $queries = [];

    /** @var boolean $enabled If Debug Stack is enabled (log queries) or not. */
    public $enabled = true;

    public $start = null;

    public $currentQuery = 0;

    /**
     * @var array
     */
    public $cachedQueries = [];

    /**
     * Constructor.
     *
     * @param LoggerInterface $logger A SfLoggerInterface instance
     */
    public function __construct(SfLoggerInterface $logger = null)
    {
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function startQuery($sql, array $params = null, array $types = null)
    {
        if ($this->enabled) {
            foreach (['_connection', '_tableName'] as $key) {
                if (isset($params[$key])) {
                    unset($params[$key]);
                }
            }

            $this->start = microtime(true);
            $this->queries[++$this->currentQuery] = ['sql' => $sql, 'params' => $params, 'types' => $types, 'executionMS' => 0];
            $this->lastExecutedQuery = $sql;
        }

        if (null !== $this->logger) {
            $this->log(substr($sql, 0, 1024));
        }
    }

    /**
     * {@inheritdoc}
     */
    public function stopQuery($cached = false)
    {
        $this->queries[$this->currentQuery]['executionMS'] = $cached ? 0 : microtime(true) - $this->start;

        if ($cached) {
            $this->cachedQueries[] = $this->queries[$this->currentQuery];
        }
    }

    /**
     * Logs a message.
     *
     * @param string $message A message to log
     */
    public function log($message)
    {
        $this->logger->debug($message);
    }

    /**
     * Logs an exception
     *
     * @param string $message A message to log
     */
    public function logException($message)
    {
        $this->logger->err($message);
    }

    /**
     * @return string
     */
    public function getLastExecutedQuery()
    {
        return $this->lastExecutedQuery;
    }
}
