<?php

namespace Overblog\OrmsBundle\Orms;

/**
 * Orms Service
 *
 * @author xavier
 */

use Symfony\Component\DependencyInjection\ContainerInterface;

use Overblog\OrmsBundle\Orms\ServiceInterface;
use Overblog\OrmsBundle\DependencyInjection\OverblogOrmsExtension as Extension;
use Overblog\OrmsBundle\Config\Object;

use Orms\Exception\Exception;
use Orms\Object\DataObject;

class Service implements ServiceInterface
{
    /**
     * Main service container instance
     * @var ContainerInterface
     */

    protected $container;

    /**
     * Name of the open connections
     * @var array
     */

    protected $connections = [];

    /**
     * Instanciate service
     *
     * @param ContainerInterface $container
     * @param array              $connections
     */

    public function __construct(
        ContainerInterface $container,
        Array $connections
    )
    {
        $this->container   = $container;
        $this->connections = $connections;
    }

    /**
     * Return if object has configuration
     *
     * @param string $object
     *
     * @return bool
     */

    public function hasObjectConfig($object)
    {
        return $this->container->has($this->getObjectConfigName($object));
    }

    /**
     * Return the object configuration name
     *
     * @param $object
     *
     * @return string
     */

    public function getObjectConfigName($object)
    {
        return sprintf(Extension::OBJECT_NAME, $object);
    }

    /**
     * Return Object for config
     *
     * @param string $object
     *
     * @return Object
     * @throws Exception
     */

    protected function getObjectConfig($object)
    {
        if (!$this->hasObjectConfig($object))
        {
            throw new Exception(sprintf('Unknown object requested : %s', $object));
        }

        return $this->container->get($this->getObjectConfigName($object));
    }

    /**
     * Try to retrieve an object instance, or instanciate it from configuration
     *
     * @param string $object
     * @param array  $param
     *
     * @return DataObject
     *
     * @throws Exception
     */

    public function getObject($object, $param = null)
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        return new $class(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );
    }

    /**
     * Return an object array based on search param
     *
     * @param string $object
     * @param array $param
     *
     * @return \ArrayIterator
     *
     * @throws Exception
     */

    public function getObjects($object, $param = [])
    {
        $config = $this->getObjectConfig($object);

        /** @var DataObject $class */

        $class  = $config->getClass();

        return $class::find(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );
    }

    /**
     * Return the first object object array based on search param
     *
     * @param string $object
     * @param array  $param
     *
     * @return DataObject
     *
     * @throws Exception
     */

    public function getFirstObject($object, $param = [])
    {
        // override the offset/limit parameters

        $param['offset'] = 0;
        $param['limit']  = 1;

        $os = $this->getObjects($object, $param);

        if (1 === $os->count())
        {
            return $os->offsetGet(0);
        }

        return null;
    }


    /**
     * Return something special based on a DataObject model, but using a self
     * defined special method.
     *
     * @param string $object
     * @param string $method
     * @param array  $param
     *
     * @return mixed
     *
     * @throws Exception
     */

    public function getFromMethod(
        $object,
        $method,
        $param = []
    )
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        return $class::$method(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );
    }

    /**
     * Returns the object count based on the parameters
     *
     * @param string $object
     * @param array $param
     *
     * @return integer
     *
     * @throws \Exception|\Orms\Exception\Exception
     */

    public function getNbObjects($object, $param = [])
    {
        try
        {
            $config = $this->getObjectConfig($object);
        }
        catch (Exception $e)
        {
            throw $e;
        }

        $class = $config->getClass();

        /** @var DataObject $class */

        return $class::findNb($config->getProvider(), $param);
    }

    /**
     * Return metadata on the last query looking for objects objects
     *
     * @param string $object
     *
     * @return mixed
     *
     * @throws Exception
     */

    public function getMetaData($object)
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        /** @var DataObject $class */

        return $class::getMetaData($config->getProvider());
    }

    /**
     * @param string $object
     */

    public function getManager($object)
    {
        static $manager;

        if (! is_null($manager))
        {
            return $manager;
        }

        $config = $this->getObjectConfig($object);
        $class  = $config->getManager();

        $manager = new $class($this);

        return $manager;
    }

    /**
     * Return container
     *
     * @return ContainerInterface
     */

    public function getContainer()
    {
        return $this->container;
    }

    /**
     * Return ids of connections instanciate
     *
     * @return Array
     */

    public function getConnections()
    {
        return $this->connections;
    }
}
