<?php
namespace AlloCine\OrmsBundle\Command;

use AlloCine\OrmsBundle\Command\BaseCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Finder\Finder;

use AlloCine\OrmsBundle\Fixtures\ObjectsInterface;
use AlloCine\OrmsBundle\Fixtures\LoaderInterface;
use Orms\Object\DataObject;

/**
 * Check fixtures objects for the specified Bundle and try to load them
 * Fixtures files must implement the AlloCine\OrmsBundle\Fixtures\FixturesInterface
 */
class DatabaseFixturesCommand extends BaseCommand implements LoaderInterface
{
    /** ORM service **/
    protected $orms;

    /** Object refereces store **/
    protected $references;

    /** Store ENV **/
    protected $env;

    protected function configure()
    {
        $this->setName('orms:fixtures:load')
          ->setDescription('Load orms fixtures from <BasePath>/<Bundle>/Fixtures/');

        $this->addOption('onlyBundle', null, InputOption::VALUE_REQUIRED, 'Run only script for this bundle');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->orms      = $this->getContainer()->get('orms');
        $this->env       = $input->getOption('env');
        $fixtures        = [];

        foreach ($this->getBundlesToRun($input) as $name => $bundle) {
            $bundlePath      = $bundle->getPath();
            $bundleNamespace = $bundle->getNamespace();

            $fixturesPath    = $bundlePath."/Fixtures/";

            if (!file_exists($fixturesPath)) {
                continue;
            }

            $previousCount = count($fixtures);

            foreach (Finder::create()->files()->name('*.php')->sortByName()->in($fixturesPath) as $fixturesFile) {
                $fixtureClass = sprintf("%s\\Fixtures\\%s", $bundleNamespace, basename($fixturesFile, '.php'));

                // Check if fixture is a class
                if (!class_exists($fixtureClass)) {
                    continue;
                }

                $fixtures[] = $fixtureClass;
            }

            if ($previousCount != count($fixtures)) {
                $output->writeln(sprintf('<info>Found bundle %s</info>', $name));
            }
        }

        foreach ($fixtures as $fixtureClass) {
            $fixturesObject = new $fixtureClass();
            $fixturesObject->setFixturesLoader($this);

            if (!$fixturesObject instanceof ObjectsInterface) {
                $output->writeln("<error> Fixture class $fixtureClass must implement the ObjectsInterface</error>");
                continue;
            }

            $this->loadFixtures($fixturesObject, $output);
        }
    }

    /**
     * load fixtures from the specified FixturesObjects
     */
    protected function loadFixtures(ObjectsInterface $fixturesObject, OutputInterface $output)
    {
        $nbObjects = 0;
        $output->writeln("<info>Loading fixtures from : ".get_class($fixturesObject)."</info>");
        $output->writeln("<info>Related ORM Object : </info><comment>".$fixturesObject->getObject()."</comment>");

        foreach ($fixturesObject->getData() as $key => $entry) {
            $object = $this->orms->getObject($fixturesObject->getObject(), new \ArrayObject($entry));
            $object->save();
            if (!is_numeric($key)) {
                $this->setFixtureObjectReference($key, $object);
            }

            $nbObjects++;
        }

        $output->writeln("<info>Number of <comment>".$fixturesObject->getObject()."</comment> loaded : <comment>".$nbObjects."</comment></info>");
    }

    /**
     * Retrieve a referenced object
     */
    public function getFixtureObjectReference($key)
    {
        if (isset($this->references[$key])) {
            return $this->references[$key];
        } else {
            return false;
        }
    }

    /**
     * Set a referenced object
     */
    public function setFixtureObjectReference($key, DataObject $val)
    {
        $this->references[$key] = $val;
    }

    /**
     * Return actual env
     *
     * @return string
     */
    public function getEnv()
    {
        return $this->env;
    }
}
