CURRENT_DIR := $(shell dirname $(realpath $(lastword $(MAKEFILE_LIST))))
GCP_LOCATION ?= europe-west1
GCP_PROJECT ?= webedia-registry
REGISTRY_LOCATIONS ?= "europe europe-west1 us-east1"
EXTRA_REGISTRY ?=

ifeq (${APP_ENV},dev)
GCLOUD_PREFIX = ${DOCKER_RUN} -e CLOUDSDK_CONFIG=${DOCKER_WEBEDIA_HOME}/gcloud/${GCP_ORG}/config \
				 -e HOME=${DOCKER_WEBEDIA_HOME}/gcloud \
				 -u ${USER_ID}:${GROUP_ID} \
				 -w /workspace \
				 google/cloud-sdk:slim
GCLOUD_FLAGS = "--no-launch-browser"
else
GCLOUD_PREFIX =
GCLOUD_FLAGS =
endif

## —— Authentification 🔧   —————————————————————————————————————————————
.PHONY: auth
auth: gcloud_login gcloud_init docker_login ## Login to Google Cloud Platform / docker  using docker container

## —— Docker authentication 🐳 ———————————————————————————————————————————————————————————————————————

.PHONY: docker_login
docker_login: ## authenticate on docker registry
	@for location in `echo ${REGISTRY_LOCATIONS}`; do \
	$(call color_text,Authenticate to registry https://$${location}-docker.pkg.dev ,32); \
	${GCLOUD_PREFIX} gcloud auth print-access-token | docker login -u oauth2accesstoken --password-stdin https://$${location}-docker.pkg.dev; \
	done;
	$(MAKE) wbd_check_update

.PHONY: docker_auth_config
docker_auth_config: var-AC_KEY_PATH ## generate DOCKER_AUTH_CONFIG without TTL from a service account key
	cat ${AC_KEY_PATH} | docker login -u _json_key --password-stdin https://${GCP_LOCATION}-docker.pkg.dev

## —— Gcloud ⛅ ———————————————————————————————————————————————————————————————————————
.PHONY: gcloud_sh
gcloud_sh: ## run a shell on the gcloud container
	${GCLOUD_PREFIX} bash

.PHONY: gcloud_init
gcloud_init: ## set gcloud project configuration
	${GCLOUD_PREFIX} gcloud config set project ${GCP_PROJECT}

.PHONY: gcloud_login
gcloud_login: force ?= false
gcloud_login: ## authenticate on GCP (use force=true parameter to force re-authentication)
	([[ "${force}" != "true" ]] && [[ $(shell ${GCLOUD_PREFIX} gcloud auth list --filter="status:ACTIVE AND account ~ '.+@((ext\.)?webedia-group\.com|corp\.webedia\.fr|webedia-test-gcp\.com)'" --format="value(account)" 2> /dev/null | grep "@" | wc -l | tr -d ' ') -ne 0 ]]) || ${GCLOUD_PREFIX} gcloud auth login --update-adc ${GCLOUD_FLAGS}

.PHONY: gcloud_AD_login
gcloud_AD_login: ## authenticate application default on GCP
	${GCLOUD_PREFIX} gcloud auth application-default login ${GCLOUD_FLAGS} --scopes=openid,https://www.googleapis.com/auth/userinfo.email,https://www.googleapis.com/auth/cloud-platform,${GCLOUD_AD_SCOPES}

.PHONY: gcloud_AD_token
gcloud_AD_token: ## generate application default authentification token
	${GCLOUD_PREFIX} gcloud auth application-default print-access-token

.PHONY: gcloud_token
gcloud_token: ## generate authentification token
	${GCLOUD_PREFIX} gcloud auth print-access-token

.PHONY: gcloud_auth_list
gcloud_auth_list: ## list gcloud authentification
	${GCLOUD_PREFIX} gcloud auth list

.PHONY: gcloud_impersonate
gcloud_impersonate: var-sa ## impersonate a service account
	${GCLOUD_PREFIX} gcloud config set auth/impersonate_service_account ${sa}

.PHONY: gcloud_clean_impersonate
gcloud_clean_impersonate: ## clear impersonate configuration
	${GCLOUD_PREFIX} gcloud config unset auth/impersonate_service_account

.PHONY: gcloud_ci_login
gcloud_ci_login: GCP_TOKEN_LIFETIME ?= 3600
gcloud_ci_login: var-GITLAB_WIF_PROVIDER var-SERVICE_ACCOUNT var-GITLAB_OIDC_TOKEN var-GOOGLE_CREDENTIALS
	@$(call color_text,GCP Token duration: ${GCP_TOKEN_LIFETIME} seconds,33)
	gcloud iam workload-identity-pools create-cred-config \
		${GITLAB_WIF_PROVIDER} \
		--service-account=${SERVICE_ACCOUNT} \
		--service-account-token-lifetime-seconds=${GCP_TOKEN_LIFETIME} \
		--output-file=${GOOGLE_CREDENTIALS} \
		--credential-source-file=.gitlab_jwt_token
	echo "${GITLAB_OIDC_TOKEN}" > .gitlab_jwt_token
	gcloud auth login --cred-file=${GOOGLE_CREDENTIALS}
	export CLOUDSDK_AUTH_ACCESS_TOKEN=`gcloud auth print-access-token`; \
	for location in `echo ${REGISTRY_LOCATIONS}`; do echo "$${CLOUDSDK_AUTH_ACCESS_TOKEN}" | docker login -u oauth2accesstoken --password-stdin https://$${location}-docker.pkg.dev; done; \
	for registry in `echo ${EXTRA_REGISTRY}`; do echo "$${CLOUDSDK_AUTH_ACCESS_TOKEN}" | docker login -u oauth2accesstoken --password-stdin https://$${registry}; done; \
	echo "DOCKER_AUTH_CONFIG=`sed -E ':a;N;$$!ba;s/\r{0,1}\n//g' ~/.docker/config.json`" >> build.env; \
	echo "CLOUDSDK_AUTH_ACCESS_TOKEN=$${CLOUDSDK_AUTH_ACCESS_TOKEN}" >> build.env
