<?php

namespace AlloCine\ImageUrlGenerator;

use AlloCine\ImageUrlGenerator\Filter\BorderFilter;
use AlloCine\ImageUrlGenerator\Filter\CropFilter;
use AlloCine\ImageUrlGenerator\Filter\FilterInterface;
use AlloCine\ImageUrlGenerator\Filter\FormatFilter;
use AlloCine\ImageUrlGenerator\Filter\OverlayFilter;
use AlloCine\ImageUrlGenerator\Filter\ResizeFilter;

class ImageUrl
{
    const GOOGLE = 'google';
    const FACEBOOK = 'facebook';
    const OTHER = 'other';

    /**
     * @var string
     */
    private $urlBase;

    /**
     * @var int
     */
    private $serverCount;

    /**
     * @var FilterInterface[]
     */
    private $filters;

    /**
     * @var string
     */
    private $path;

    /**
     * @var string
     */
    private $domain;

    /**
     * @var bool
     */
    private $isSecure;

    /**
     * @param $path
     * @param string $urlBase
     * @param int $serverCount
     * @param bool $isSecure
     */
    public function __construct($path, $urlBase, $serverCount, $isSecure = false)
    {
        $this->domain = $this->findDomain($path);

        if (SELF::OTHER === $this->domain) {
            $parts = parse_url($path); // handle absolute or relative urls.
            $path = $parts['path'] . (array_key_exists('query', $parts) ? '?' . $parts['query'] : '');
        }

        $this->path = $path;
        $this->urlBase = $urlBase;
        $this->serverCount = $serverCount;
        $this->isSecure = $isSecure;
        $this->filters = [];
    }

    /**
     * @return FilterInterface[]
     */
    public function getFilters()
    {
        return $this->filters;
    }

    /**
     * @param FilterInterface[] $filters
     *
     * @return $this
     */
    public function setFilters(array $filters)
    {
        $this->filters = $filters;
        $this->sortFilters();

        return $this;
    }

    /**
     * @param FilterInterface $filter
     *
     * @return ImageUrl
     */
    public function addFilter(FilterInterface $filter): ImageUrl
    {
        $this->filters[] = $filter;
        $this->sortFilters();

        return $this;
    }

    /**
     * @return string
     */
    private function getScheme()
    {
        return $this->isSecure ? 'https' : 'http';
    }

    /**
     * @return string
     */
    public function getDomain()
    {
        return $this->domain;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        if (SELF::OTHER !== $this->domain) {
            return $this->path;
        }

        return $this->getUrl();
    }

    /**
     * @return string
     */
    public function getUrl()
    {
        if (! empty($this->filters)) {
            return sprintf(
                "%s://%s/%s/%s",
                $this->getScheme(),
                $this->getHost(),
                trim(implode('/', $this->filters), '/'),
                trim($this->path, '/')
            );
        } else {
            return sprintf(
                "%s://%s/%s",
                $this->getScheme(),
                $this->getHost(),
                trim($this->path, '/')
            );
        }
    }

    /**
     * @return string
     */
    public function getHost()
    {
        /** Determine the right server for this image */
        if (empty($this->filters)) {
            $path = $this->path;
        } else {
            $path = sprintf(
                "%s/%s",
                implode('/', $this->filters),
                $this->path
            );
        }

        $total = 0;

        foreach (str_split($path) as $character) {
            $total += ord($character);
        }

        return sprintf(
            '%s.web.img%d.acsta.net',
            $this->urlBase,
            ($total % $this->serverCount ? $total % $this->serverCount + 1 : $this->serverCount)
        );
    }

    /**
     * @param string $path
     *
     * @return string
     */
    private function findDomain($path)
    {
        if (preg_match('/.*facebook.*/', $path)) {
            return SELF::FACEBOOK;
        } elseif (preg_match('/.*google.*/', $path)) {
            return SELF::GOOGLE;
        }

        return SELF::OTHER;
    }

    /**
     * @return ImageUrl
     */
    private function sortFilters(): ImageUrl
    {
        usort($this->filters, function (FilterInterface $a, FilterInterface $b) {
            return $a->getPriority() <=> $b->getPriority();
        });

        return $this;
    }


    /**
     * @param int $width
     * @param int $height
     *
     * @return ImageUrl
     */
    public function resize($width, $height): ImageUrl
    {
        return $this->addFilter(new ResizeFilter($width, $height));
    }

    /**
     * @param int $width
     * @param int $height
     * @param int $centerWidth
     * @param int $centerHeight
     *
     * @return ImageUrl
     */
    public function crop(int $width, int $height, int $centerWidth = CropFilter::DEFAULT_CENTER, int $centerHeight = CropFilter::DEFAULT_CENTER): ImageUrl
    {
        return $this->addFilter(new CropFilter($width, $height, $centerWidth, $centerHeight));
    }

    /**
     * @param int    $width
     * @param string $color
     *
     * @return $this
     */
    public function border(int $width = null, string $color = null): ImageUrl
    {
        if ($width <= 0) {
            return $this;
        }

        return $this->addFilter(new BorderFilter($width, $color));
    }

    /**
     * @param $filename
     * @param int $offset
     * @param string $position
     *
     * @return $this
     */
    public function overlay(string $filename, int $offset, string $position): ImageUrl
    {
        return $this->addFilter(new OverlayFilter($filename, $offset, $position));
    }

    /**
     * @param string $format
     *
     * @return ImageUrl
     */
    public function format(string $format): ImageUrl
    {
        return $this->addFilter(new FormatFilter($format));
    }
}
