<?php

namespace AlloCine\ImageUrlGenerator\Filter;

use AlloCine\ImageUrlGenerator\Exception\ConfigurationException;

class CropFilter implements FilterInterface
{
    const DEFAULT_CENTER = 50;

    /**
     * @var int
     */
    private $width;

    /**
     * @var int
     */
    private $height;

    /**
     * @var int
     */
    private $centerWidth;

    /**
     * @var int
     */
    private $centerHeight;

    /**
     * @param int|null $width
     * @param int|null $height
     * @param int      $centerWidth
     * @param int      $centerHeight
     *
     * @throws ConfigurationException
     */
    public function __construct(
        int $width = null,
        int $height = null,
        int $centerWidth = self::DEFAULT_CENTER,
        int $centerHeight = self::DEFAULT_CENTER
    ) {
        if ($width === null && $height === null) {
            throw new ConfigurationException('Crop error. You must provide "width" or "height".');
        }

        $this->setWidth($width);
        $this->setHeight($height);
        $this->setCenterHeight($centerHeight);
        $this->setCenterWidth($centerWidth);
    }

    /**
     * @return int
     */
    public function getWidth(): int
    {
        return $this->width;
    }

    /**
     * @param int $width
     *
     * @return CropFilter
     */
    public function setWidth(int $width): CropFilter
    {
        if (!is_numeric($width)) {
            throw new \InvalidArgumentException('Width must be numeric');
        }

        $this->width = $width;

        return $this;
    }

    /**
     * @return int
     */
    public function getHeight(): int
    {
        return $this->height;
    }

    /**
     * @param $height
     *
     * @return CropFilter
     *
     * @throws ConfigurationException
     */
    public function setHeight(int $height): CropFilter
    {
        $this->height = $height;

        return $this;
    }

    /**
     * @return int
     */

    public function getCenterWidth(): int
    {
        return $this->centerWidth;
    }

    /**
     * @param $centerWidth
     *
     * @return CropFilter
     *
     * @throws ConfigurationException
     */
    public function setCenterWidth(int $centerWidth): CropFilter
    {
        if ($this->centerWidth < 0 || $this->centerWidth > 100) {
            throw new ConfigurationException(
                sprintf(
                    'Crop error. "centerWidth" must be between 0 and 100. ' .
                    '"%s" given',
                    $this->centerHeight
                )
            );
        }

        $this->centerWidth = $centerWidth;

        return $this;
    }

    /**
     * @return int
     */
    public function getCenterHeight(): int
    {
        return $this->centerHeight;
    }

    /**
     * @param $centerHeight
     *
     * @return CropFilter
     *
     * @throws ConfigurationException
     */
    public function setCenterHeight(int $centerHeight): CropFilter
    {
        if ($this->centerHeight < 0 || $this->centerHeight > 100) {
            throw new ConfigurationException(
                sprintf(
                    'Crop error. "centerHeight" must be between 0 and 100. ' .
                    '"%s" given',
                    $this->centerHeight
                )
            );
        }

        $this->centerHeight = $centerHeight;

        return $this;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        if ((self::DEFAULT_CENTER === $this->centerHeight)
            &&
            (self::DEFAULT_CENTER === $this->centerWidth)
        ) {
            return sprintf('c_%s_%s', $this->width, $this->height);
        } else {
            return sprintf(
                'c_%s_%s_%s_%s',
                $this->width,
                $this->height,
                $this->centerWidth,
                $this->centerHeight
            );
        }
    }

    /**
     * @return int
     */

    public function getPriority(): int
    {
        return 20;
    }
}
