<?php

namespace AlloCine\ImageUrlGenerator\Filter;

use AlloCine\ImageUrlGenerator\Exception\ConfigurationException;

class BorderFilter implements FilterInterface
{
    /**
     * @var int
     */
    private $width;

    /**
     * @var int
     */
    private $color;

    /**
     * @param int    $width
     * @param string $color
     */
    public function __construct(int $width, string $color = 'ffffff')
    {
        $this->setWidth($width);
        $this->setColor($color);
    }

    /**
     * @return int
     */
    public function getWidth(): int
    {
        return $this->width;
    }

    /**
     * @param int $width
     *
     * @return BorderFilter
     */
    public function setWidth(int $width): BorderFilter
    {
        if (!is_numeric($width)) {
            throw new \InvalidArgumentException('Width must be numeric');
        }

        $this->width = $width;

        return $this;
    }

    /**
     * @return int
     */
    public function getColor(): string
    {
        return $this->color;
    }

    /**
     * @param $color
     *
     * @return BorderFilter
     *
     * @throws ConfigurationException
     */
    public function setColor(string $color): BorderFilter
    {
        $color = strtolower($color);

        if (!preg_match('/[a-f0-9]{6}/', $color)) {
            throw new ConfigurationException(sprintf(
                'Border error. "color" must be a valid hexadecimal color. "%s" given',
                $color
            ));
        }

        $this->color = $color;

        return $this;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return sprintf('b_%s_%s', $this->width, $this->color);
    }

    /**
     * @return int
     */
    public function getPriority(): int
    {
        return 30;
    }
}
