<?php

namespace Tests\AlloCine\ImageUrlGenerator\Filter;

use AlloCine\ImageUrlGenerator\BrandConfig;
use AlloCine\ImageUrlGenerator\Filter\BorderFilter;
use AlloCine\ImageUrlGenerator\Filter\CropFilter;
use AlloCine\ImageUrlGenerator\Filter\FormatFilter;
use AlloCine\ImageUrlGenerator\Filter\ResizeFilter;
use AlloCine\ImageUrlGenerator\ImageUrl;

class ImageUrlTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @return array
     */
    public function getHostProvider()
    {
        return [
            [
                '/commons/empty.gif',
                'fr',
                6,
                'fr.web.img4.acsta.net',
                []
            ],
            [
                '/commons/webui/shared/loader_blue.gif',
                'fr',
                6,
                'fr.web.img6.acsta.net',
                []
            ],
            [
                '/commons/webui/FR/misc/billeterie.png',
                'fr',
                6,
                'fr.web.img2.acsta.net',
                []
            ],
            [
                '/medias/nmedia/18/65/73/07/19130288.jpg',
                'fr',
                6,
                'fr.web.img3.acsta.net',
                [
                    new ResizeFilter(160, 240),
                    new BorderFilter(1, 'd6d6d6')
                ]
            ],
            [
                '/newsv7/15/07/20/12/31/260214.jpg',
                'fr',
                6,
                'fr.web.img6.acsta.net',
                [
                    new ResizeFilter(288, 115)
                ]
            ],
            [
                '/medias/nmedia/18/65/73/07/19130302.jpg',
                'fr',
                6,
                'fr.web.img6.acsta.net',
                [
                    new CropFilter(100, 100),
                    new BorderFilter(1, 'd6d6d6')
                ]
            ],
            [
                '/commons/emptymedia/avatar_default.jpg',
                'fr',
                6,
                'fr.web.img5.acsta.net',
                [
                    new CropFilter(75, 80)
                ]
            ],
            [
                '/videothumbnails/15/05/13/12/30/567425.jpg',
                'fr',
                6,
                'fr.web.img6.acsta.net',
                [
                    new ResizeFilter(160, 240)
                ]
            ],
            [
                'videothumbnails/15/05/13/12/30/567425.jpg',
                'fr',
                6,
                'fr.web.img3.acsta.net',
                [
                    new FormatFilter('png')
                ]
            ]
        ];
    }


    /**
     * @dataProvider getHostProvider
     *
     * @param $path
     * @param $urlBase
     * @param $serverCount
     * @param $expected
     * @param $filters
     */
    public function testGetHost(
        $path,
        $urlBase,
        $serverCount,
        $expected,
        $filters
    ) {
        $imageUrl = (new ImageUrl($path, $urlBase, $serverCount))
            ->setFilters($filters);

        $this->assertEquals($expected, $imageUrl->getHost());
    }

    public function testFluentInterface()
    {
        $imageUrl = new ImageUrl('/newsv7/15/07/20/12/31/260214.jpg', 'fr', 6);

        $imageUrl
            ->resize(150, 150)
            ->crop(150, 150)
            ->border(2, 'FF0000')
            ->overlay('club300a.png', 0, 'se')
            ->format('png')
        ;

        $this->assertSame(
            'http://fr.web.img3.acsta.net/f_png/r_150_150/c_150_150/b_2_ff0000/o_club300a.png_0_se/newsv7/15/07/20/12/31/260214.jpg',
            (string) $imageUrl
        );
    }

    public function testConfig()
    {
        $imageUrl = ImageUrl::fromConfig('foo.jpg', BrandConfig::ALLOCINE);

        $this->assertSame('http://fr.web.img2.acsta.net/foo.jpg', (string) $imageUrl);
    }
}
