<?php

namespace AlloCine\ImageUrlGenerator;

use AlloCine\ImageUrlGenerator\Filter\BorderFilter;
use AlloCine\ImageUrlGenerator\Filter\CropFilter;
use AlloCine\ImageUrlGenerator\Filter\FilterInterface;
use AlloCine\ImageUrlGenerator\Filter\FormatFilter;
use AlloCine\ImageUrlGenerator\Filter\OverlayFilter;
use AlloCine\ImageUrlGenerator\Filter\ResizeFilter;

class ImageUrl
{
    const GOOGLE = 'google';
    const FACEBOOK = 'facebook';
    const OTHER = 'other';

    private string $urlBase;

    private int $serverCount;

    /**
     * @var FilterInterface[]
     */
    private array $filters;

    private string $path;

    private string $domain;

    private bool $isSecure;

    private string $hostPattern;

    public function __construct(string $path, ConfigImageGeneratorInterface $config, bool $isSecure = false)
    {
        $this->domain = $this->findDomain($path);

        if (self::OTHER === $this->domain) {
            $parts = parse_url($path); // handle absolute or relative urls.
            $path = $parts['path'] . (array_key_exists('query', $parts) ? '?' . $parts['query'] : '');
        }

        $this->path = $path;
        $this->hostPattern = $config->getHostPattern();
        $this->urlBase = $config->getPrefix();
        $this->serverCount = $config->getServerCount();
        $this->isSecure = $isSecure;
        $this->filters = [];
    }

    public static function fromConfig(string $path, ConfigImageGeneratorInterface $config, bool $isSecure = false): self
    {
        return new ImageUrl($path, $config, $isSecure);
    }

    public function getFilters(): array
    {
        return $this->filters;
    }

    public function setFilters(array $filters): self
    {
        $this->filters = $filters;
        $this->sortFilters();

        return $this;
    }

    public function addFilter(FilterInterface $filter): self
    {
        $this->filters[] = $filter;
        $this->sortFilters();

        return $this;
    }

    private function getScheme(): string
    {
        return $this->isSecure() ? 'https' : 'http';
    }

    public function getDomain(): string
    {
        return $this->domain;
    }

    public function __toString(): string
    {
        if (self::OTHER !== $this->domain) {
            return $this->path;
        }

        return $this->getUrl();
    }

    public function isSecure(): bool
    {
        return $this->isSecure;
    }

    public function getUrl(): string
    {
        if (!empty($this->filters)) {
            return sprintf(
                "%s://%s/%s/%s",
                $this->getScheme(),
                $this->getHost(),
                trim(implode('/', $this->filters), '/'),
                trim($this->path, '/')
            );
        }

        return sprintf(
            "%s://%s/%s",
            $this->getScheme(),
            $this->getHost(),
            trim($this->path, '/')
        );
    }

    public function getHost(): string
    {
        /** Determine the right server for this image */
        if (empty($this->filters)) {
            $path = $this->path;
        } else {
            $path = sprintf(
                "%s/%s",
                implode('/', $this->filters),
                $this->path
            );
        }

        $total = 0;

        foreach (str_split($path) as $character) {
            $total += ord($character);
        }

        return sprintf(
            $this->hostPattern,
            $this->urlBase,
            ($total % $this->serverCount ? $total % $this->serverCount + 1 : $this->serverCount)
        );
    }

    private function findDomain(string $path): string
    {
        if (preg_match('/.*facebook.*/', $path)) {
            return self::FACEBOOK;
        }

        if (preg_match('/.*google.*/', $path)) {
            return self::GOOGLE;
        }

        return self::OTHER;
    }

    private function sortFilters(): self
    {
        usort($this->filters, fn(FilterInterface $a, FilterInterface $b) => $a->getPriority() <=> $b->getPriority());

        return $this;
    }


    public function resize(string $width, string $height): self
    {
        return $this->addFilter(new ResizeFilter($width, $height));
    }

    public function crop(int $width, int $height, int $centerWidth = CropFilter::DEFAULT_CENTER_WIDTH, int $centerHeight = CropFilter::DEFAULT_CENTER_HEIGHT): self
    {
        return $this->addFilter(new CropFilter($width, $height, $centerWidth, $centerHeight));
    }

    public function border(?int $width = null, ?string $color = null): self
    {
        if ($width <= 0) {
            return $this;
        }

        return $this->addFilter(new BorderFilter($width, $color));
    }

    public function overlay(string $filename, int $offset, string $position): self
    {
        return $this->addFilter(new OverlayFilter($filename, $offset, $position));
    }

    public function format(string $format): self
    {
        return $this->addFilter(new FormatFilter($format));
    }
}
