<?php

namespace AlloCine\ImageUrlGenerator\Filter;

use AlloCine\ImageUrlGenerator\Exception\ConfigurationException;

class CropFilter implements FilterInterface
{
    const DEFAULT_CENTER_WIDTH = 50;
    const DEFAULT_CENTER_HEIGHT = 0;

    private int $width;

    private int $height;

    private int $centerWidth;

    private int $centerHeight;

    public function __construct(
        ?int $width = null,
        ?int $height = null,
        int $centerWidth = self::DEFAULT_CENTER_WIDTH,
        int $centerHeight = self::DEFAULT_CENTER_HEIGHT
    ) {
        if ($width === null && $height === null) {
            throw new ConfigurationException('Crop error. You must provide "width" or "height".');
        }

        $this->setWidth($width);
        $this->setHeight($height);
        $this->setCenterHeight($centerHeight);
        $this->setCenterWidth($centerWidth);
    }

    public function getWidth(): int
    {
        return $this->width;
    }

    public function setWidth(int $width): CropFilter
    {
        if (!is_numeric($width)) {
            throw new \InvalidArgumentException('Width must be numeric');
        }

        $this->width = $width;

        return $this;
    }

    public function getHeight(): int
    {
        return $this->height;
    }

    public function setHeight(int $height): CropFilter
    {
        $this->height = $height;

        return $this;
    }

    public function getCenterWidth(): int
    {
        return $this->centerWidth;
    }

    public function setCenterWidth(int $centerWidth): CropFilter
    {
        if ($centerWidth < 0 || $centerWidth > 100) {
            throw new ConfigurationException(
                sprintf(
                    'Crop error. "centerWidth" must be between 0 and 100. ' .
                    '"%s" given',
                    $centerWidth
                )
            );
        }

        $this->centerWidth = $centerWidth;

        return $this;
    }

    public function getCenterHeight(): int
    {
        return $this->centerHeight;
    }

    public function setCenterHeight(int $centerHeight): CropFilter
    {
        if ($centerHeight < 0 || $centerHeight > 100) {
            throw new ConfigurationException(
                sprintf(
                    'Crop error. "centerHeight" must be between 0 and 100. ' .
                    '"%s" given',
                    $centerHeight
                )
            );
        }

        $this->centerHeight = $centerHeight;

        return $this;
    }

    public function __toString(): string
    {
        if ((self::DEFAULT_CENTER_HEIGHT === $this->centerHeight)
            &&
            (self::DEFAULT_CENTER_WIDTH === $this->centerWidth)
        ) {
            return sprintf('c_%s_%s', $this->width, $this->height);
        }

        return sprintf(
            'c_%s_%s_%s_%s',
            $this->width,
            $this->height,
            $this->centerWidth,
            $this->centerHeight
        );
    }

    public function getPriority(): int
    {
        return 20;
    }
}
