<?php

namespace AlloCine\ImageUrlGenerator\Filter;

use AlloCine\ImageUrlGenerator\Exception\ConfigurationException;

class ResizeFilter implements FilterInterface
{
    const EMPTY_VALUE = 'x';

    /**
     * @var string
     */
    private $width;

    /**
     * @var string
     */
    private $height;

    /**
     * @param string|null $width
     * @param string|null $height
     *
     * @throws ConfigurationException
     */
    public function __construct(string $width = null, string $height = null)
    {
        if ($width === null && $height === null) {
            throw new ConfigurationException('Resize error. You must provide "width" or "height".');
        }

        $this->setWidth($width ?: self::EMPTY_VALUE);
        $this->setHeight($height ?: self::EMPTY_VALUE);
    }

    /**
     * @return string
     */
    public function getWidth(): string
    {
        return $this->width;
    }

    /**
     * @param string $width
     *
     * @return ResizeFilter
     */
    public function setWidth(string $width): ResizeFilter
    {
        if (!is_numeric($width) && $width !== self::EMPTY_VALUE) {
            throw new \InvalidArgumentException(
                sprintf('Width must be numeric or an empty value symbol ("%d").', self::EMPTY_VALUE)
            );
        }

        $this->width = $width;

        return $this;
    }

    /**
     * @return string
     */
    public function getHeight(): string
    {
        return $this->height;
    }

    /**
     * @param $height
     *
     * @return ResizeFilter
     */
    public function setHeight(string $height): ResizeFilter
    {
        if (!is_numeric($height) && $height !== self::EMPTY_VALUE) {
            throw new \InvalidArgumentException(
                sprintf('Height must be numeric or an empty value symbol ("%d").', self::EMPTY_VALUE)
            );
        }

        $this->height = $height;

        return $this;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return sprintf('r_%s_%s', $this->width, $this->height);
    }

    /**
     * @return int
     */
    public function getPriority(): int
    {
        return 10;
    }
}
