<?php

declare(strict_types=1);

namespace AlloCine\I18NBundle\Translation;

use Exception;
use GuzzleHttp\Command\Exception\CommandClientException;
use GuzzleHttp\Command\Result;
use Loco\Http\ApiClient;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\HttpFoundation\Response;

class Loco
{
    private ApiClient $client;

    public function __construct(string $apiKey)
    {
        $this->client = ApiClient::factory(['key' => $apiKey, 'request.options' => ['verify' => false]]);
    }

    public function export(string $locale, string $tag): string
    {
        $result = $this->client->exportLocale([
            'locale' => $locale,
            'ext' => 'xlf',
            'format' => 'symfony',
            'filter' => $tag,
            'status' => 'all',
        ]);

        return LocoFormatter::unformat($result->__toString());
    }

    /**
     * @throws CommandClientException
     * @throws \Exception
     */
    public function import(
        SplFileInfo $file,
        string $locale,
        ?string $tag = null,
        bool $source = false,
    ): void {
        $this->client->import([
            'ext' => 'xlf',
            'src' => LocoFormatter::format($file->getContents()),
            'async' => false,
            'locale' => $locale,
            'index' => ($source ? 'text' : 'id'),
            'tag-new' => $tag,
            'tag-updated' => $tag,
        ]);

        /* Suppression du tag "Import batch X" */
        $this->deleteTag('Import batch 1 (xlf via API)');
    }

    public function getTags(): Result
    {
        return $this->client->getTags();
    }

    public function getLocales(): array
    {
        $result = [];

        foreach ($this->client->getLocales() as $locale) {
            $result[$locale['code']] = $locale['code'];
        }

        return $result;
    }

    /**
     * Delete a tag but ignore exception if tag doesn't exist.
     *
     * @throws CommandClientException
     * @throws \Exception
     */
    public function deleteTag($tag): Result|array
    {
        try {
            return $this->client->deleteTag(['tag' => $tag]);
        } catch (CommandClientException $e) {
            if (Response::HTTP_NOT_FOUND !== $e->getResponse()->getStatusCode()) {
                throw $e;
            }
        }

        return [];
    }

    /**
     * Delete a translation.
     *
     * @throws CommandClientException
     */
    public function deleteTranslation(string $locale, string $key): Result|array
    {
        try {
            return $this->client->untranslate([
                'locale' => $locale,
                'id' => $key,
            ]);
        } catch (CommandClientException $e) {
            if (Response::HTTP_NOT_FOUND !== $e->getResponse()->getStatusCode()) {
                throw $e;
            }
        }

        return [];
    }

    /**
     * Create a new translation.
     */
    public function addTranslation(string $locale, string $key, string $translation, bool $checkIfExists = true): Result
    {
        if ($checkIfExists) {
            try {
                $this->getAsset($key);
            } catch (CommandClientException $e) {
                if (Response::HTTP_NOT_FOUND === $e->getResponse()->getStatusCode()) {
                    $this->createAsset($key);
                } else {
                    throw $e;
                }
            }
        }

        $params = [
            'locale' => $locale,
            'id' => $key,
            'translation' => $translation,
        ];

        return $this->client->translate($params);
    }

    /**
     * Return Asset configuration for a given Key.
     */
    public function getAsset(string $key): Result
    {
        return $this->client->getAsset(['id' => $key]);
    }

    /**
     * Create an asset for a given key.
     */
    public function createAsset(string $key, string $default = 'untranslated'): Result
    {
        return $this->client->createAsset([
            'id' => $key,
            'name' => $key,
            'type' => 'text',
            'default' => $default,
        ]);
    }

    /**
     * Delete an asset for a given key.
     */
    public function deleteAsset(string $key): Result
    {
        return $this->client->deleteAsset(['id' => $key]);
    }

    /**
     * Tags an asset with a new or existing term. If the tag doesn't exist it will be created.
     */
    public function tagAsset(string $key, string $tag): Result
    {
        return $this->client->tagAsset(['id' => $key, 'name' => $tag]);
    }
}
