<?php

declare(strict_types=1);

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Extractor\AdvancedExtractor;
use AlloCine\I18NBundle\Translation\Manager;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Translation\Catalogue\MergeOperation;
use Symfony\Component\Translation\MessageCatalogueInterface;
use Symfony\Component\Translation\Writer\TranslationWriter;

#[AsCommand(
    name: 'translation:update:all',
    description: 'Crawls the entire project and dumps xlf translation files.',
)]
class UpdateAllCommand extends Command
{
    public function __construct(private readonly ParameterBagInterface $parameterBag, private readonly TranslationWriter $writer, private readonly Manager $manager, private readonly AdvancedExtractor $advancedExtractor)
    {
        parent::__construct();
    }

    public function execute(InputInterface $input, OutputInterface $output): int
    {
        $paths = $this->parameterBag->get('translation.source.paths');
        $locales = $this->parameterBag->get('translation.ui.locales');

        $output->writeln('Merging translation keys for known brands.');

        foreach ($locales as $locale) {
            $this->doExecute($output, $locale, $paths);
        }

        $output->writeln('<info>Merge complete.</info>');

        return Command::SUCCESS;
    }

    private function doExecute(OutputInterface $output, string $locale, array $paths): void
    {
        $path = $this->parameterBag->get('translation.ui.path');
        $extractedCatalogue = $this->advancedExtractor->extract($locale, $paths);
        $currentCatalogue = $this->manager->getCatalogue($locale);

        $merge = new MergeOperation($currentCatalogue, $extractedCatalogue);

        $all = $new = 0;
        foreach ($merge->getDomains() as $domain) {
            $intlDomain = $domain.MessageCatalogueInterface::INTL_DOMAIN_SUFFIX;
            $newMessages = $merge->getNewMessages($domain);

            if ([] === $newMessages || ([] === $currentCatalogue->all($intlDomain) && [] !== $currentCatalogue->all($domain))) {
                continue;
            }

            $result = $merge->getResult();
            $allIntlMessages = $result->all($intlDomain);
            $currentMessages = array_diff_key($newMessages, $result->all($domain));
            $result->replace($currentMessages, $domain);
            $result->replace($allIntlMessages + $newMessages, $intlDomain);

            $new += \count($newMessages);
            $all += \count($allIntlMessages);
        }

        $output->writeln(\sprintf('Found %s (%d) keys (<info>%d newly added</info>).', $locale, $all, $new));

        $this->writer->write($merge->getResult(), 'xlf', [
            'path' => $path,
            'default_locale' => $this->parameterBag->get('translation.default.locale'),
        ]);
    }
}
