<?php

declare(strict_types=1);

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Translation\Loco;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\File\File;

#[AsCommand(
    name: 'translation:loco:export',
    description: 'Export all translation from Loco',
)]
class LocoExportCommand extends Command
{
    public function __construct(private readonly ParameterBagInterface $parameterBag, private readonly Loco $loco)
    {
        parent::__construct();
    }

    public function configure(): void
    {
        $this
            ->addOption(
                'icu_format',
                null,
                InputOption::VALUE_OPTIONAL,
                'Use ICU messages format.',
                true
            )
        ;
    }

    public function execute(InputInterface $input, OutputInterface $output): int
    {
        $output = new SymfonyStyle($input, $output);
        $path = $this->parameterBag->get('translation.ui.path');

        $tags = $this->loco->getTags();
        $locales = $this->loco->getLocales();

        $icuFormat = (bool) $input->getOption('icu_format');

        foreach ($locales as $locale => $realLocale) {
            foreach ($tags as $tag) {
                $export = $this->loco->export($realLocale, $tag);

                try {
                    $filePattern = $icuFormat ? '%s/%s+intl-icu.%s.xlf' : '%s/%s.%s.xlf';
                    $filename = \sprintf($filePattern, $path, $tag, $locale);
                    $file = new File($filename);
                } catch (\Exception) {
                    $filename = \sprintf('%s/%s.%s.xlf', $path, $tag, $realLocale);
                    $file = new File($filename);
                }

                $output->text($filename);

                $file->openFile('w')->fwrite($export);
            }
        }

        return Command::SUCCESS;
    }
}
