<?php

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Extractor\AdvancedExtractor;
use AlloCine\I18NBundle\Translation\Manager;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Translation\Catalogue\TargetOperation;
use Symfony\Component\Translation\Writer\TranslationWriter;

class FindObsoletesCommand extends ContainerAwareCommand
{
    /**
     * @var TranslationWriter
     */
    private $writer;

    /**
     * @var Manager
     */
    private $manager;

    /**
     * @var AdvancedExtractor
     */
    private $advancedExtractor;

    public function __construct(TranslationWriter $writer, Manager $manager, AdvancedExtractor $advancedExtractor)
    {
        $this->writer = $writer;
        $this->manager = $manager;
        $this->advancedExtractor = $advancedExtractor;
        parent::__construct();
    }

    /**
     * {@inheritsDoc}
     */
    public function configure()
    {
        $this
            ->setName('translation:obsoletes:find')
            ->setDescription('Crawls the entire project and displays unused placeholders.');
    }

    /**
     * {@inheritsDoc}
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {
        $paths = $this->getContainer()->getParameter('translation.source.paths');
        $locales = $this->getContainer()->getParameter('translation.ui.locales');

        foreach ($locales as $locale) {
            $this->doExecute($output, $locale, $paths);
        }
    }

    /**
     * @param string   $locale
     * @param string[] $paths
     */
    private function doExecute(OutputInterface $output, $locale, $paths)
    {
        $extractedCatalogue = $this->advancedExtractor->extract($locale, $paths);

        $currentCatalogue = $this->manager->getCatalogue($locale);

        $diff = new TargetOperation($currentCatalogue, $extractedCatalogue);

        $output->writeln(sprintf('Obsolete "<comment>%s</comment>" keys found:', $locale));
        $count = 0;

        foreach ($diff->getDomains() as $domain) {
            $messages = $diff->getObsoleteMessages($domain);
            $count += count($messages);

            if (count($messages)) {
                $output->writeln(sprintf('<info>Domain "%s"</info>:', $domain));
                foreach (array_keys($messages) as $key) {
                    $output->writeln(sprintf(' - %s', $key));
                }
            }
        }

        $output->writeln(sprintf('<comment>%d</comment> obsolete keys found.', $count));
        $output->writeln('');
    }
}
