<?php

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Translation\Manager;
use AppBundle\Console\AllocineStyle;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use AlloCine\I18NBundle\Model\TranslationMessage;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Translation\Writer\TranslationWriter;

class AddCommand extends Command
{
    private const ALL_LOCALES = "ALL LOCALES";
    private const DEFAULT_TRANS_PREFIX = "missing__";

    /**
     * @var AllocineStyle
     */
    private $io;

    /**
     * @var TranslationWriter
     */
    private $writer;

    /**
     * @var ParameterBagInterface
     */
    private $parameterBag;

    /**
     * @var Manager
     */
    private $manager;

    public function __construct(ParameterBagInterface $parameterBag, TranslationWriter $writer, Manager $manager)
    {
        $this->parameterBag = $parameterBag;
        $this->writer = $writer;
        $this->manager = $manager;
        parent::__construct();
    }

    /**
     * {@inheritsDoc}
     */
    public function configure()
    {
        $this
            ->setName('translation:add')
            ->setDescription('Add a new translation string.')
        ;
    }

    /**
     * {@inheritsDoc}
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->io = new AllocineStyle($input, $output);
        $defaultLocale = $this->parameterBag->get('translation.defaut.locale');

        $locales = $this->manager->getLocales();
        asort($locales);
        $locales = array_values($locales);
        $localesChoices = array_merge([self::ALL_LOCALES], $locales);

        $domain = $this->io->autocomplete('Please enter the domain of the translation', 'messages', null, $this->manager->getAvailableDomains());
        $key    = $this->io->ask('Please enter the key of the translation');
        $value  = $this->io->ask(sprintf('Please enter the "%s" translation', $defaultLocale));

        $message = new TranslationMessage($key, $value);
        $this->manager->saveDomain($this->manager->getCatalogue($defaultLocale)->getDomain($domain)->add($message));

        while ($this->io->confirm('Do you wish to add another translation for this key ?', false)) {
            $selectedLocale = $this->io->choice('Please choose a locale', $localesChoices, $defaultLocale);

            if (self::ALL_LOCALES === $selectedLocale) {
                foreach ($locales as $locale) {
                    $this->addLocaleTranslation($locale, $domain, $key);
                }
            } else {
                $this->addLocaleTranslation($selectedLocale, $domain, $key);
            }
        }
    }

    private function addLocaleTranslation(string $locale, string $domain, string $key)
    {
        $defaultValue = self::DEFAULT_TRANS_PREFIX . $key;
        $value = $this->io->ask(sprintf('Please enter the "%s" translation', $locale), $defaultValue);
        $this->manager->saveDomain($this->manager->getCatalogue($locale)->getDomain($domain)->add(new TranslationMessage($key, $value)));
    }
}
