<?php

namespace AlloCine\I18NBundle\Translation;

use Symfony\Bundle\FrameworkBundle\Translation\TranslationLoader;
use Symfony\Component\Translation\Reader\TranslationReader;
use Symfony\Component\Translation\Writer\TranslationWriter;
use AlloCine\I18NBundle\Model\TranslationCatalogue;
use AlloCine\I18NBundle\Model\TranslationDomain;

class Manager
{
    /**
     * @var TranslationReader
     */
    private $reader;

    /**
     * @var TranslationWriter
     */
    private $writer;

    /**
     * @var string
     */
    private $path;

    /**
     * @var string[]
     */
    private $locales;

    /**
     * @var string
     */
    private $default_locale;

    /**
     * @var TranslationCatalogue[]
     */
    private $catalogues = [];

    /**
     * @param TranslationReader $reader
     * @param TranslationWriter $writer
     * @param string            $path
     * @param string[]          $locales
     * @param string            $default_locale
     */
    public function __construct(
        TranslationReader $reader,
        TranslationWriter $writer,
        $path,
        array $locales,
        $default_locale
    )
    {
        $this->reader         = $reader;
        $this->writer         = $writer;
        $this->path           = $path;
        $this->locales        = $locales;
        $this->default_locale = $default_locale;
    }

    /**
     * @return string[]
     */
    public function getLocales()
    {
        return $this->locales;
    }

    /**
     * @return string[]
     */
    public function getAvailableDomains()
    {
        $domains = [];

        foreach ($this->getCatalogues() as $catalogue) {
            $domains = array_merge($domains, $catalogue->getDomains());
        }

        return array_unique($domains);
    }

    /**
     * @return TranslationCatalogue[]
     */
    public function getCatalogues()
    {
        $catalogues = [];

        foreach ($this->locales as $locale) {
            $catalogues[]= $this->getCatalogue($locale);
        }

        return $catalogues;
    }

    /**
     * @param TranslationDomain $domain
     *
     * @return Manager
     */
    public function saveDomain(TranslationDomain $domain)
    {
        $domain->getSource()->addDomain($domain);
        $this->writer->write($domain->getSource(), 'xlf', [
            'path' => $this->path,
            'default_locale' => $this->default_locale
        ]);

        return $this;
    }

    /**
     * @param string $locale
     *
     * @return TranslationCatalogue
     */
    public function getCatalogue($locale)
    {
        if (!array_key_exists($locale, $this->catalogues)) {
            $this->catalogues[$locale] = new TranslationCatalogue($locale);
            $this->reader->read($this->path, $this->catalogues[$locale]);
        }

        return $this->catalogues[$locale];
    }
}
