<?php

declare(strict_types=1);

namespace AlloCine\I18NBundle\Translation;

use AlloCine\I18NBundle\Model\TranslationCatalogue;
use AlloCine\I18NBundle\Model\TranslationDomain;
use Symfony\Component\Translation\Reader\TranslationReader;
use Symfony\Component\Translation\Writer\TranslationWriter;

class Manager
{
    private array $catalogues = [];

    public function __construct(
        private readonly TranslationReader $reader,
        private readonly TranslationWriter $writer,
        private $path,
        private readonly array $locales,
        private $defaultLocale,
    ) {
    }

    public function getLocales(): array
    {
        return $this->locales;
    }

    public function getAvailableDomains(): array
    {
        $domains = [];

        foreach ($this->getCatalogues() as $catalogue) {
            $domains = array_merge($domains, $catalogue->getDomains());
        }

        return array_unique($domains);
    }

    public function getCatalogues(): array
    {
        $catalogues = [];

        foreach ($this->locales as $locale) {
            $catalogues[] = $this->getCatalogue($locale);
        }

        return $catalogues;
    }

    public function saveDomain(TranslationDomain $domain): static
    {
        $domain->getSource()->addDomain($domain);
        $this->writer->write($domain->getSource(), 'xlf', [
            'path' => $this->path,
            'default_locale' => $this->defaultLocale,
        ]);

        return $this;
    }

    public function getCatalogue(string $locale): TranslationCatalogue
    {
        if (!\array_key_exists($locale, $this->catalogues)) {
            $this->catalogues[$locale] = new TranslationCatalogue($locale);
            $this->reader->read($this->path, $this->catalogues[$locale]);
        }

        return $this->catalogues[$locale];
    }
}
