<?php

declare(strict_types=1);

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Extractor\AdvancedExtractor;
use AlloCine\I18NBundle\Translation\Manager;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Translation\Catalogue\TargetOperation;

#[AsCommand(
    name: 'translation:obsoletes:find',
    description: 'Crawls the entire project and displays unused placeholders.',
)]
class FindObsoletesCommand extends Command
{
    public function __construct(
        private readonly ParameterBagInterface $parameterBag,
        private readonly Manager $manager,
        private readonly AdvancedExtractor $advancedExtractor,
    ) {
        parent::__construct();
    }

    public function execute(InputInterface $input, OutputInterface $output): int
    {
        $paths = $this->parameterBag->get('translation.source.paths');
        $locales = $this->parameterBag->get('translation.ui.locales');

        foreach ($locales as $locale) {
            $this->doExecute($output, $locale, $paths);
        }

        return Command::SUCCESS;
    }

    private function doExecute(OutputInterface $output, string $locale, array $paths): void
    {
        $extractedCatalogue = $this->advancedExtractor->extract($locale, $paths);
        $currentCatalogue = $this->manager->getCatalogue($locale);

        $diff = new TargetOperation($currentCatalogue, $extractedCatalogue);

        $output->writeln(\sprintf('Obsolete "<comment>%s</comment>" keys found:', $locale));
        $count = 0;

        foreach ($diff->getDomains() as $domain) {
            $messages = $diff->getObsoleteMessages($domain);
            $count += \count($messages);

            if (\count($messages)) {
                $output->writeln(\sprintf('<info>Domain "%s"</info>:', $domain));
                foreach (array_keys($messages) as $key) {
                    $output->writeln(\sprintf(' - %s', $key));
                }
            }
        }

        $output->writeln(\sprintf('<comment>%d</comment> obsolete keys found.', $count));
        $output->writeln('');
    }
}
