<?php

declare(strict_types=1);

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Command\Style\CustomStyle;
use AlloCine\I18NBundle\Model\TranslationMessage;
use AlloCine\I18NBundle\Translation\Manager;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;

#[AsCommand(
    name: 'translation:add',
    description: 'Add a new translation string.',
)]
class AddCommand extends Command
{
    private const string ALL_LOCALES = 'ALL LOCALES';
    private const string DEFAULT_TRANS_PREFIX = 'missing__';

    private SymfonyStyle $io;

    public function __construct(private readonly ParameterBagInterface $parameterBag, private readonly Manager $manager)
    {
        parent::__construct();
    }

    public function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->io = new CustomStyle($input, $output);
        $defaultLocale = $this->parameterBag->get('translation.default.locale');

        $locales = $this->manager->getLocales();
        asort($locales);
        $locales = array_values($locales);
        $localesChoices = array_merge([self::ALL_LOCALES], $locales);

        $domain = $this->io->autocomplete('Please enter the domain of the translation', 'messages', null, $this->manager->getAvailableDomains());
        $key = $this->io->ask('Please enter the key of the translation');
        $value = $this->io->ask(\sprintf('Please enter the "%s" translation', $defaultLocale));

        $message = new TranslationMessage($key, $value);
        $this->manager->saveDomain($this->manager->getCatalogue($defaultLocale)->getDomain($domain)->add($message));

        while ($this->io->confirm('Do you wish to add another translation for this key ?', false)) {
            $selectedLocale = $this->io->choice('Please choose a locale', $localesChoices, $defaultLocale);

            if (self::ALL_LOCALES === $selectedLocale) {
                foreach ($locales as $locale) {
                    $this->addLocaleTranslation($locale, $domain, $key);
                }
            } else {
                $this->addLocaleTranslation($selectedLocale, $domain, $key);
            }
        }

        return Command::SUCCESS;
    }

    private function addLocaleTranslation(string $locale, string $domain, string $key): void
    {
        $defaultValue = self::DEFAULT_TRANS_PREFIX.$key;
        $value = $this->io->ask(\sprintf('Please enter the "%s" translation', $locale), $defaultValue);
        $this->manager->saveDomain($this->manager->getCatalogue($locale)->getDomain($domain)->add(new TranslationMessage($key, $value)));
    }
}
