## —— Boundary 🔐️  ———————————————————————————————————————————————————————————————————————
export BOUNDARY_IMAGE = europe-docker.pkg.dev/prj-shd-prd-registry-8ed4/foundation/boundary-client:0.19
export BOUNDARY_ADDR = https://boundary.webedia-group.net

BOUNDARY_PREFIX = ${DOCKER_RUN} --rm -ti \
	-e ALL_PROXY=socks5://localhost:1080 \
	-e BOUNDARY_ADDR=${BOUNDARY_ADDR} \
	-e GOOGLE_APPLICATION_CREDENTIALS=${ADC_FILE_DOCKER} \
	-e SKIP_SETCAP=true \
	--network host \
	-u ${USER_ID}:${GROUP_ID} \
	${BOUNDARY_IMAGE}

BOUNDARY_BIN ?= boundary
BOUNDARY_AUTH_FILE_HOST = ${WEBEDIA_HOME}/boundary/.boundary-auth.json
BOUNDARY_AUTH_FILE_DOCKER = ${DOCKER_WEBEDIA_HOME}/boundary/.boundary-auth.json

BOUNDARY_TOKEN_FILE_HOST =  ${WEBEDIA_HOME}/boundary/.boundary-token
export BOUNDARY_TOKEN_FILE_DOCKER =  ${DOCKER_WEBEDIA_HOME}/boundary/.boundary-token
BOUNDARY_TOKEN_ARG=-token "file://${BOUNDARY_TOKEN_FILE_DOCKER}"
CLOUDSQL_PROXY_BASE = ${BOUNDARY_PREFIX} ${BOUNDARY_BIN} connect -target-name=socks-proxy -listen-port=1080 -target-scope-name=common-${env} ${BOUNDARY_TOKEN_ARG} -exec cloud-sql-proxy --
ESCLOUD_DOMAINS = europe-west3.gcp.cloud.es.io europe-west1.gcp.cloud.es.io europe-west9.gcp.elastic-cloud.com
ESCLOUD_ENDPOINTS = es kb
ESCLOUD_NGINX_CONFIG_FILE = ${WEBEDIA_HOME}/boundary/escloud_proxy_nginx_config.conf
ESCLOUD_REGION_PORT_MAP = europe-west3:9243 europe-west1:9244 europe-west9:9245

# Function to return the appropriate target scope argument
define get_target_scope_arg
$(if $(2),\
	-target-scope-name=$(2),\
	-target-scope-id=$(shell ${BOUNDARY_PREFIX} sh -c "${BOUNDARY_BIN} targets list -recursive ${BOUNDARY_TOKEN_ARG} -format json | jq -r '.items[] | select(.name==\"$(1)\") | .scope_id'"))
endef

# Function to get the port for a given region
define get_escloud_region_port
$(shell echo ${ESCLOUD_REGION_PORT_MAP} | tr ' ' '\n' | grep ${1} | cut -d':' -f2)
endef

.PHONY: boundary_token_file
boundary_token_file:
	@${BOUNDARY_PREFIX} sh -c "cat ${BOUNDARY_AUTH_FILE_DOCKER} | sed -e 's!http[s]\?://\S*!!g' | jq -r .item.attributes.token > ${BOUNDARY_TOKEN_FILE_DOCKER}"

.PHONY: boundary_login
boundary_login: force ?= false
boundary_login: ## login to Boundary (force=true to force login)
	@([[ "${force}" != "true" ]] && [ -f "${BOUNDARY_TOKEN_FILE_HOST}" ] && ${BOUNDARY_PREFIX} ${BOUNDARY_BIN} auth-tokens list ${BOUNDARY_TOKEN_ARG} >/dev/null 2>&1) \
	|| (${BOUNDARY_PREFIX} sh -c 'rm -f ${BOUNDARY_AUTH_FILE_DOCKER} && ${BOUNDARY_BIN} authenticate oidc -keyring-type=none -format=json | tee -a ${BOUNDARY_AUTH_FILE_DOCKER}' \
	&& $(MAKE) boundary_token_file)

.PHONY: boundary_logout
boundary_logout: ## logout from Boundary
	@rm -f ${BOUNDARY_AUTH_FILE_HOST} ${BOUNDARY_TOKEN_FILE_HOST}

.PHONY: boundary_sh
boundary_sh: ## open a shell into Boundary container
	@${BOUNDARY_PREFIX} sh

.PHONY: boundary_connect
boundary_connect: team_env ?=
boundary_connect: var-target ## start a tcp session to a target (target, team_env (optional), port (optional))
boundary_connect: boundary_login
	@${BOUNDARY_PREFIX} ${BOUNDARY_BIN} connect -target-name=${target} -listen-addr=0.0.0.0 $(if ${port},-listen-port ${port},) $(call get_target_scope_arg,${target},${team_env}) ${BOUNDARY_TOKEN_ARG}

.PHONY: boundary_connect_pg
boundary_connect_pg: team_env ?=
boundary_connect_pg: db ?= postgres
boundary_connect_pg: var-target var-user ## connect to a postgresql database (instance, user, db (optional), team_env (optional))
boundary_connect_pg: boundary_login
	@${BOUNDARY_PREFIX} ${BOUNDARY_BIN} connect postgres -target-name=${target} $(call get_target_scope_arg,${target},${team_env}) -username=${user} -dbname=${db} ${BOUNDARY_TOKEN_ARG}

.PHONY: boundary_connect_mysql
boundary_connect_mysql: team_env ?=
boundary_connect_mysql: port ?= 53306
boundary_connect_mysql: var-target var-user ## connect to a mysql database (instance, user, port (optional))
boundary_connect_mysql: boundary_login
	@${BOUNDARY_PREFIX} ${BOUNDARY_BIN} connect -target-name=${target} -listen-port=${port} $(call get_target_scope_arg,${target},${team_env}) ${BOUNDARY_TOKEN_ARG} -exec \
	mysql -- -h 127.0.0.1 -P ${port} -u ${user} -p

.PHONY: boundary_connect_redis
boundary_connect_redis: team_env ?=
boundary_connect_redis: port ?= 56379
boundary_connect_redis: var-target ## start a tcp session to a target (target, team_env (optional), port (optional))
boundary_connect_redis: boundary_login
	@${BOUNDARY_PREFIX} ${BOUNDARY_BIN} connect -target-name=${target} -listen-port ${port} $(call get_target_scope_arg,${target},${team_env}) ${BOUNDARY_TOKEN_ARG} -exec \
	redis-cli -- -p ${port}

.PHONY: boundary_connect_pg_iam
boundary_connect_pg_iam: port ?= 5432
boundary_connect_pg_iam: var-connection_name var-env ## start a socks proxy to a cloudsql instance (connection_name, env, port (optional))
boundary_connect_pg_iam: boundary_login
	${CLOUDSQL_PROXY_BASE} ${connection_name} --private-ip --address 0.0.0.0 --port ${port} --auto-iam-authn

.PHONY: boundary_connect_mysql_iam
boundary_connect_mysql_iam: port ?= 3306
boundary_connect_mysql_iam: var-connection_name var-env ## start a socks proxy to a cloudsql instance (connection_name, env, port (optional))
boundary_connect_mysql_iam: boundary_login
	@${CLOUDSQL_PROXY_BASE} ${connection_name} --private-ip --address 0.0.0.0 --port ${port} --auto-iam-authn

.PHONY: boundary_target_list
boundary_target_list: ## list all boundary targets
boundary_target_list: boundary_login
	@${BOUNDARY_PREFIX} ${BOUNDARY_BIN} targets list -recursive ${BOUNDARY_TOKEN_ARG}

.PHONY: boundary_start_escloud_tunnel
boundary_start_escloud_tunnel: var-env var-region
boundary_start_escloud_tunnel: boundary_login
boundary_start_escloud_tunnel: ## start a tcp session to a target (env, region)
	@${BOUNDARY_PREFIX} ${BOUNDARY_BIN} connect -target-name=escloud-${region} -listen-port=$(call get_escloud_region_port,${region}) -target-scope-name=common-${env} ${BOUNDARY_TOKEN_ARG}

.PHONY: generate_escloud_certificates
generate_escloud_certificates: mkcert_init ## generate certificates for all Elastic Cloud domain and endpoints
	@for domain in ${ESCLOUD_DOMAINS}; do \
		for endpoint in ${ESCLOUD_ENDPOINTS}; do \
			host=*.$${endpoint}.$${domain}; \
			[[ -f ${WEBEDIA_HOME}/certs/$${host}.crt ]] || mkcert -cert-file ${WEBEDIA_HOME}/certs/$$host.crt -key-file ${WEBEDIA_HOME}/certs/$$host.key $$host; \
		done; \
	done

.PHONY: boundary_start_escloud_proxy
boundary_start_escloud_proxy:
boundary_start_escloud_proxy: boundary_login generate_escloud_certificates ## connect to an elasticsearch cluster
	## Generate the nginx configuration file that will handle all ESCloud regions & endpoints
	@echo "" > $(ESCLOUD_NGINX_CONFIG_FILE)
	@$(foreach domain,$(ESCLOUD_DOMAINS),$(foreach endpoint,$(ESCLOUD_ENDPOINTS), \
	ESCLOUD_TUNNEL_PORT=$(call get_escloud_region_port,$$(echo $(domain) | cut -d'.' -f1)); \
	printf "\
		server {\n \
			listen 9243 ssl; listen 443 ssl; \
			server_name *.$(endpoint).$(domain); \
			ssl_protocols TLSv1 TLSv1.1 TLSv1.2 TLSv1.3; \
			ssl_certificate /etc/nginx/certs/*.$(endpoint).$(domain).crt; \
			ssl_certificate_key /etc/nginx/certs/*.$(endpoint).$(domain).key; \
			location / { \
				proxy_pass https://127.0.0.1:$${ESCLOUD_TUNNEL_PORT}; \
				proxy_set_header Host \$$host; \
			}\
		}" \
	 >> $(ESCLOUD_NGINX_CONFIG_FILE);) )

	@${DOCKER_RUN} --net host \
	-v ${ESCLOUD_NGINX_CONFIG_FILE}:/etc/nginx/conf.d/default.conf \
	-v ${WEBEDIA_HOME}/certs:/etc/nginx/certs \
	europe-docker.pkg.dev/prj-shd-prd-registry-8ed4/foundation/nginx:1.27-alpine
