<?php

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Extractor\AdvancedExtractor;
use AlloCine\I18NBundle\Translation\Manager;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Translation\Catalogue\MergeOperation;
use Symfony\Component\Translation\MessageCatalogueInterface;
use Symfony\Component\Translation\Writer\TranslationWriter;

class UpdateAllCommand extends Command
{
    /**
     * @var TranslationWriter
     */
    private $writer;

    /**
     * @var Manager
     */
    private $manager;

    /**
     * @var ParameterBagInterface
     */
    private $parameterBag;

    /**
     * @var AdvancedExtractor
     */
    private $advancedExtractor;

    public function __construct(ParameterBagInterface $parameterBag, TranslationWriter $writer, Manager $manager, AdvancedExtractor $advancedExtractor)
    {
        $this->parameterBag = $parameterBag;
        $this->writer = $writer;
        $this->manager = $manager;
        $this->advancedExtractor = $advancedExtractor;
        parent::__construct();
    }

    /**
     * {@inheritsDoc}
     */
    public function configure()
    {
        $this
            ->setName('translation:update:all')
            ->setDescription('Crawls the entire project and dumps xlf translation files.')
        ;
    }

    /**
     * {@inheritsDoc}
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {

        $paths = $this->parameterBag->get('translation.source.paths');
        $locales = $this->parameterBag->get('translation.ui.locales');

        $output->writeln(sprintf('Merging translation keys for known brands.'));

        foreach ($locales as $locale) {
            $this->doExecute($output, $locale, $paths);
        }

        $output->writeln(sprintf('<info>Merge complete.</info>'));

        return 0;
    }

    /**
     * @param string   $locale
     */
    private function doExecute(OutputInterface $output, $locale, array $paths)
    {
        $path   = $this->parameterBag->get('translation.ui.path');

        $extractedCatalogue = $this->advancedExtractor->extract($locale, $paths);

        $currentCatalogue = $this->manager->getCatalogue($locale);

        $merge = new MergeOperation($currentCatalogue, $extractedCatalogue);

        $all = $new = 0;
        foreach ($merge->getDomains() as $domain) {
            $intlDomain = $domain.MessageCatalogueInterface::INTL_DOMAIN_SUFFIX;
            $newMessages = $merge->getNewMessages($domain);

            if ([] === $newMessages || ([] === $currentCatalogue->all($intlDomain) && [] !== $currentCatalogue->all($domain))) {
                continue;
            }

            $result = $merge->getResult();
            $allIntlMessages = $result->all($intlDomain);
            $currentMessages = array_diff_key($newMessages, $result->all($domain));
            $result->replace($currentMessages, $domain);
            $result->replace($allIntlMessages + $newMessages, $intlDomain);

            $new += count($newMessages);
            $all += count($allIntlMessages);
        }

        $output->writeln(sprintf('Found %s (%s) keys (<info>%d newly added</info>).', $locale, $all, $new));

        $this->writer->write($merge->getResult(), 'xlf', [
            'path'           => $path,
            'default_locale' => $this->parameterBag->get('translation.defaut.locale')
        ]);
    }
}
