<?php

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Translation\Loco;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Translation\Loader\XliffFileLoader;
use Symfony\Component\Translation\Catalogue\TargetOperation;
use SebastianBergmann\Diff\Parser;
use AlloCine\I18NBundle\Translation\LocoFormatter;
use GuzzleHttp\Command\Exception\CommandClientException;
use Symfony\Component\Translation\Exception\NotFoundResourceException;

/**
 * Description of LocoImportDiffCommand
 *
 * @author Xavier HAUSHERR <xhausherr@allocine.fr>
 */
class LocoImportDiffCommand extends Command
{
    /**
     * @var Loco
     */
    private $loco;

    /**
     * @var ParameterBagInterface
     */
    private $parameterBag;

    /**
     * @var KernelInterface
     */
    private $kernel;

    public function __construct(ParameterBagInterface $parameterBag, Loco $loco, KernelInterface $kernel)
    {
        $this->parameterBag = $parameterBag;
        $this->loco = $loco;
        $this->kernel = $kernel;
        parent::__construct();
    }

    /**
     * {@inheritsDoc}
     */
    public function configure()
    {
        $this
            ->setName('translation:loco:import:diff')
            ->setDescription('Import diff translation from xlf files to Loco.')
            ->addArgument('commit', InputArgument::REQUIRED, 'Actual Commit')
            ->addArgument('prevCommit', InputArgument::REQUIRED, 'Previous commit to compare');
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {
        $output = new SymfonyStyle($input, $output);
        $path   = $this->parameterBag->get('translation.ui.path');

        $commit = $input->getArgument('commit');
        $prevCommit = $input->getArgument('prevCommit');
        $xliffLoader = new XliffFileLoader();
        $assetsToClean = [];

        $output->title(sprintf('Getting diff between %s ... %s', $prevCommit, $commit));

        $diffList = $this->getDiff($commit, $prevCommit, $path);

        foreach ($diffList as $diff) {
            $to = str_replace('b/', '', $diff->getTo());
            $match = [];

            if (!preg_match('#([^\+]+)(\+intl-icu)?\.([a-z]{2}_[A-Z]{2})\.xlf$#', basename($to), $match)) {
                continue;
            }

            $locale = $match[3];
            $domain = $match[1];

            $tmFile = tempnam($this->kernel->getCacheDir(), 'tr8n');
            file_put_contents(
                $tmFile,
                $this->getFileFromCommit($prevCommit, $to)
            );

            $sourceFile = $xliffLoader->load($tmFile, $locale, $domain);

            try {
                $targetFile = $xliffLoader->load(
                    $this->kernel->getRootDir() . '/../' . $to,
                    $locale,
                    $domain
                );
            } catch (NotFoundResourceException $e) {
                $output->warning($e->getMessage());
                continue;
            }

            $targetOperation = new TargetOperation($sourceFile, $targetFile);
            $obsolete = $new = [];

            foreach ($targetOperation->getObsoleteMessages($domain) as $key => $translation) {
                $obsolete[] = $assetsToClean[] = $key;
            }

            foreach ($targetOperation->getNewMessages($domain) as $key => $translation) {
                if (strpos($translation, LocoFormatter::PREFIX) === false) {
                    $this->loco->addTranslation($locale, $key, $translation);
                } else {
                    // Just check if asset is created and create it if needed
                    try {
                        $this->loco->createAsset($key);
                    } catch (CommandClientException $e) {
                        // Only throw if it's different from conflict
                        if (409 != $e->getResponse()->getStatusCode()) {
                            throw $e;
                        }
                    } finally {
                        // Create tag in every case
                        $this->loco->tagAsset($key, $domain);
                    }
                }

                $new[] = sprintf('%s => %s (%s)', $key, $translation, $domain);
            }

            if (count($obsolete) > 0 or count($new) > 0) {
                $output->section($to);

                if (count($obsolete) > 0) {
                    $output->text('Remove obsolete message');
                    $output->listing($obsolete);
                }

                if (count($new) > 0) {
                    $output->text('Adding new message');
                    $output->listing($new);
                }
            }
        }

        // Clean empty asset
        foreach (array_unique($assetsToClean) as $key) {
            try {
                $asset = $this->loco->getAsset($key);
            } catch (CommandClientException $e) {
                continue;
            }

            $this->loco->deleteAsset($key);
            $output->success(sprintf('Clean obsolete key: %s', $key));
        }

        return 0;
    }

    /**
     * Get DIFF between two commits
     * @param string $commit
     * @param string $prevCommit
     * @param string $path
     * @return array
     * @throws ProcessFailedException
     */
    private function getDiff($commit, $prevCommit, $path)
    {
        $parser = new Parser;

        $process = new Process(['git', 'diff', $commit, $prevCommit, $path]);
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        return $parser->parse($process->getOutput());
    }

    /**
     * Get git version of a file
     * @param string $commit
     * @param string $file
     * @return string
     * @throws ProcessFailedException
     */
    private function getFileFromCommit($commit, $file)
    {
        $process = new Process(['git', 'show', sprintf('%s:%s', $commit, $file)]);
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        return $process->getOutput();
    }
}
