<?php

namespace AlloCine\I18NBundle\Command;

use AlloCine\I18NBundle\Extractor\AdvancedExtractor;
use AlloCine\I18NBundle\Translation\Manager;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Translation\Catalogue\MergeOperation;
use Symfony\Component\Translation\Writer\TranslationWriter;

class UpdateAllCommand extends ContainerAwareCommand
{
    /**
     * @var TranslationWriter
     */
    private $writer;

    /**
     * @var Manager
     */
    private $manager;

    /**
     * @var AdvancedExtractor
     */
    private $advancedExtractor;

    public function __construct(TranslationWriter $writer, Manager $manager, AdvancedExtractor $advancedExtractor)
    {
        $this->writer = $writer;
        $this->manager = $manager;
        $this->advancedExtractor = $advancedExtractor;
        parent::__construct();
    }

    /**
     * {@inheritsDoc}
     */
    public function configure()
    {
        $this
            ->setName('translation:update:all')
            ->setDescription('Crawls the entire project and dumps xlf translation files.')
        ;
    }

    /**
     * {@inheritsDoc}
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {

        $paths = $this->getContainer()->getParameter('translation.source.paths');
        $locales = $this->getContainer()->getParameter('translation.ui.locales');

        $output->writeln(sprintf('Merging translation keys for known brands.'));

        foreach ($locales as $locale) {
            $this->doExecute($output, $locale, $paths);
        }

        $output->writeln(sprintf('<info>Merge complete.</info>'));
    }

    /**
     * @param string   $locale
     */
    private function doExecute(OutputInterface $output, $locale, array $paths)
    {
        $path   = $this->getContainer()->getParameter('translation.ui.path');

        $extractedCatalogue = $this->advancedExtractor->extract($locale, $paths);

        $currentCatalogue = $this->manager->getCatalogue($locale);

        $merge = new MergeOperation($currentCatalogue, $extractedCatalogue);

        $all = $new = 0;
        foreach ($merge->getDomains() as $domain) {
            $new += count($merge->getNewMessages($domain));
            $all += count($merge->getMessages($domain));
        }

        $output->writeln(sprintf('Found %s (%s) keys (<info>%d newly added</info>).', $locale, $all, $new));

        $this->writer->disableBackup();
        $this->writer->write($merge->getResult(), 'xlf', [
            'path'           => $path,
            'default_locale' => $this->getContainer()->getParameter('translation.defaut.locale')
        ]);
    }
}
