<?php
/**
 * Created by PhpStorm.
 * User: yannick
 * Date: 15/09/2015
 * Time: 08:00
 */

namespace AlloCine\I18NBundle\Translation;

use AlloCine\I18NBundle\Translation\LocoFormatter;
use GuzzleHttp\Command\Exception\CommandClientException;
use Loco\Http\ApiClient;
use Loco\Http\Response\RawResponse;
use Symfony\Component\Finder\SplFileInfo;

/**
 * Class Loco
 * @package AlloCine\I18NBundle\Translation
 */

class Loco
{
    /**
     * @var ApiClient $client
     */
    private $client;

    /**
     * Loco constructor.
     *
     * @param $apiKey
     */
    public function __construct($apiKey)
    {
        $this->client = ApiClient::factory([ 'key' => $apiKey, 'request.options' => ['verify' => false ]]);
    }

    /**
     * @param string $locale
     * @param string $tag
     * @return string
     */
    public function export($locale, $tag)
    {
        /** @var RawResponse $r */

        $result = $this->client->exportLocale([
            'locale' => $locale,
            'ext'    => 'xlf',
            'format' => 'symfony',
            'filter' => $tag,
            'status' => 'all'
        ]);

        return LocoFormatter::unformat($result->__toString());
    }

    /**
     * @param SplFileInfo $file
     * @param $locale
     * @param null $tag
     * @param bool $source
     * @throws CommandClientException
     * @throws \Exception
     */
    public function import(
        SplFileInfo $file,
        $locale,
        $tag = null,
        $source = false
    ) {
        $this->client->import([
            'ext'         => 'xlf',
            'src'         => LocoFormatter::format($file->getContents()),
            'async'       => false,
            'locale'      => $locale,
            'index'       => ($source ? 'text' : 'id'),
            'tag-new'     => $tag,
            'tag-updated' => $tag,
        ]);

        /** Suppression du tag "Import batch X" */
        $this->deleteTag('Import batch 1 (xlf via API)');
    }

    /**
     * @return string[]
     */
    public function getTags()
    {
        return $this->client->getTags();
    }

    /**
     * @return string[]
     */
    public function getLocales()
    {
        $result = [];

        foreach ($this->client->getLocales() as $locale) {
            $result[$locale['code']] = $locale['code'];
        }

        return $result;
    }

    /**
     * Delete a tag but ignore exception if tag doesn't exist
     *
     * @param $tag
     * @return array
     * @throws CommandClientException
     * @throws \Exception
     */
    public function deleteTag($tag)
    {
        try {
            return $this->client->deleteTag(['tag' => $tag]);
        } catch (CommandClientException $e) {
            if (404 !== $e->getResponse()->getStatusCode()) {
                throw $e;
            }
        }

        return [];
    }

    /**
     * Delete a translation
     *
     * @param string $locale
     * @param string $key
     * @return array
     * @throws CommandClientException
     */
    public function deleteTranslation($locale, $key)
    {
        try {
            return $this->client->untranslate([
                'locale' => $locale,
                'id' => $key
            ]);
        } catch (CommandClientException $e) {
            if (404 !== $e->getResponse()->getStatusCode()) {
                throw $e;
            }
        }

        return [];
    }

    /**
     * Create a new translation
     *
     * @param string  $locale
     * @param string  $key
     * @param string  $translation
     * @param boolean $checkifExists
     * @return \GuzzleHttp\Command\Result
     */
    public function addTranslation($locale, $key, $translation, $checkifExists = true)
    {
        if ($checkifExists) {
            try {
                $this->getAsset($key);
            } catch (CommandClientException $e) {
                if (404 == $e->getResponse()->getStatusCode()) {
                    $this->createAsset($key);
                } else {
                    throw $e;
                }
            }
        }

        $params = [
            'locale' => $locale,
            'id' => $key
        ];

        if (!is_null($translation)) {
            $params['translation'] = $translation;
        }

        return $this->client->translate($params);
    }

    /**
     * Return Asset configuration for a given Key
     *
     * @param string $key
     * @return \GuzzleHttp\Command\Result
     */
    public function getAsset($key)
    {
        return $this->client->getAsset(['id' => $key]);
    }

    /**
     * Create an asset for a given key
     *
     * @param string $key
     * @param string $default
     * @return \GuzzleHttp\Command\Result
     */
    public function createAsset($key, $default = 'untranslated')
    {
        return $this->client->createAsset([
            'id' => $key,
            'name' => $key,
            'type' => 'text',
            'default' => $default
        ]);
    }

    /**
     * Delete an asset for a given key
     *
     * @param string $key
     * @return \GuzzleHttp\Command\Result
     */
    public function deleteAsset($key)
    {
        return $this->client->deleteAsset(['id' => $key]);
    }

    /**
     * Tags an asset with a new or existing term. If the tag doesn't exist it will be created.
     *
     * @param string $key
     * @param string $tag
     * @return \GuzzleHttp\Command\Result
     */
    public function tagAsset($key, $tag)
    {
        return $this->client->tagAsset(['id' => $key, 'name' => $tag]);
    }
}
