<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use AlloCine\GraphClient\Bundle\Exception\FileNotFoundException;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class GraphClient
{
    /**
     * @var GraphApiBridge
     */
    protected $bridge;

    /**
     * @var GraphCursor
     */
    private $cursor;

    /**
     * @var QueryParser
     */
    private $parser;

    /**
     * @var PhpArrayAdapter
     */
    private $warmUp;

    /**
     * @var string
     */
    private $query;

    /**
     * @var string
     */
    private $queryName;

    /**
     * @var string|null
     */
    private $preparedQuery = null;

    /**
     * @var array
     */
    private $variables = [];

    /**
     * @var string
     */
    private $uri;

    /**
     * GraphClient constructor.
     *
     * @param GraphApiBridge  $bridge
     * @param GraphCursor     $cursor
     * @param QueryParser     $parser
     * @param PhpArrayAdapter $warmUp
     */
    public function __construct(
        GraphApiBridge $bridge,
        GraphCursor $cursor,
        QueryParser $parser,
        PhpArrayAdapter $warmUp
    ) {
        $this->bridge = $bridge;
        $this->cursor = $cursor;
        $this->parser = $parser;
        $this->warmUp = $warmUp;
    }

    public function getConfig(): array
    {
        return [
            'apiHost' => $this->bridge->getApiHost(),
            'apiUri' => $this->bridge->getApiUri(),
            'apiCountryName' => $this->bridge->getApiCountryName(),
        ];
    }

    /**
     * @param string $query
     */
    private function parseQuery(string $query): void
    {
        if (!empty($this->variables)) {
            $variables = sprintf(', "variables": %s', \json_encode($this->variables));
        } else {
            $variables = '';
        }

        $this->query = \sprintf(
            '{"query": "%s"%s}',
            addslashes(preg_replace('@\s+@', ' ', $query)),
            $variables
        );
    }

    private function init(): void
    {
        $this->query = null;
        $this->preparedQuery = null;
        $this->uri = null;
        $this->variables = [];
        $this->queryName = null;
    }

    protected function computeHash()
    {
        return \md5(\sprintf('%s_%s_%s', $this->uri, $this->queryName, \serialize($this->variables)));
    }

    /**
     * Compute hash for testing purpose (builing and find fixtures)
     * @return string
     */
    protected function computeTestHash()
    {
        $variables = $this->variables;
        array_walk_recursive($variables,
            function (&$value, $key) {
                if (is_string($value) && 0 < preg_match('#^[0-9]{4}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])$#', $value)) {
                    $value = null;
                }
            }
        );

        return \md5(\sprintf('%s_%s_%s', $this->uri, $this->queryName, \serialize($variables)));
    }

    /**
     * @return string
     */
    public function getQuery(): string
    {
        if (null === $this->query) {
            return $this->preparedQuery;
        }

        return $this->query;
    }

    /**
     * @param string $uri
     *
     * @return self
     */
    public function setUri(string $uri): GraphClient
    {
        $this->uri = $uri;

        return $this;
    }

    /**
     * @param array $variables
     *
     * @return self
     */
    public function setVariables(array $variables): GraphClient
    {
        $this->variables = $variables;

        return $this;
    }

    /**
     * @param int $cursor
     * @param int $offset
     *
     * @return null|string
     */
    public function getAfterCursor(int $cursor, int $offset = 0)
    {
        return $this->cursor->getAfterCursor($cursor, $offset);
    }

    /**
     * @param string $query
     *
     * @return GraphClient
     * @throws FileNotFoundException
     * @throws \Psr\Cache\InvalidArgumentException
     */
    public function prepare(string $query): GraphClient
    {
        $this->preparedQuery = $this->parser
            ->setFragments(
                $this->warmUp
                    ->getItem(GraphApiCacheWarmer::CACHED_FRAGMENT_KEY)
                    ->get()
            )
            ->parseQuery($query);

        return $this;
    }

    /**
     * @param string|null $queryName
     *
     * @return GraphClient
     * @throws \Psr\Cache\InvalidArgumentException
     * @throws FileNotFoundException
     */
    public function generateQuery(string $queryName = null): GraphClient
    {
        $this->queryName = $queryName;
        $query = $queryName ? $this->warmUp->getItem($queryName)->get() : null;
        if (null === $this->preparedQuery && null === $query) {
            throw new FileNotFoundException("The graph query file $queryName does not exist");
        }

        if (null !== $this->preparedQuery) {
            $query = $this->preparedQuery;
        }
        $this->parseQuery($query);

        return $this;
    }

    /**
     * @return \stdClass
     * @throws BadResponseException
     */
    public function getResults(): ?\stdClass
    {
        $result = $this->bridge->query($this->query, $this->computeHash());
        $this->init();

        return $result;
    }
}
