<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use GuzzleHttp\Client;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class GraphApiBridge
{
    const TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const EXPIRATION_OFFSET = 3600;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $token;

    /**
     * @var string
     */
    private $apiHost;

    /**
     * @var string
     */
    private $apiUri;

    /**
     * @var string
     */
    private $apiCountryName;

    /**
     * @var array
     */
    private $headers;

    /**
     * @var string
     */
    private $env;

    /**
     * @var GraphApiLogger
     */
    private $logger;

    /**
     * @var GraphCoverage
     */
    private $coverage;

    /**
     * @var float
     */
    private $connectTimeout;

    /**
     * @var float
     */
    private $timeout;

    /**
     * @return string
     */
    public function getApiHost(): string
    {
        return $this->apiHost;
    }

    /**
     * @return string
     */
    public function getApiUri(): string
    {
        return $this->apiUri;
    }

    /**
     * @return string
     */
    public function getApiCountryName(): string
    {
        return $this->apiCountryName;
    }

    /**
     * @param Client         $client
     * @param GraphApiLogger $logger
     * @param GraphCoverage  $coverage
     * @param string         $env
     * @param string         $token
     * @param string         $apiHost
     * @param string         $apiUri
     * @param string         $apiCountryName
     * @param int            $connectTimeout
     * @param array          $headers
     */
    public function __construct(
        $client,
        GraphApiLogger $logger,
        GraphCoverage $coverage,
        string $env,
        string $token,
        string $apiHost,
        string $apiUri,
        string $apiCountryName,
        float $connectTimeout = 1.5,
        float $timeout = 5,
        array $headers = []
    ) {
        $this->client = $client;
        $this->logger = $logger;
        $this->coverage = $coverage;
        $this->env = $env;
        $this->token = $token;
        $this->apiHost = $apiHost;
        $this->apiUri = $apiUri;
        $this->apiCountryName = $apiCountryName;
        $this->connectTimeout = $connectTimeout;
        $this->timeout = $timeout;
        $this->headers = $headers;
        $this->apiUrl = $apiHost . $apiUri;
    }

    /**
     * @param string $body
     * @param string $hash
     * @param string $token
     *
     * @throws BadResponseException If the Graph API returns at least one error.
     *
     * @return Object
     */
    public function queryAsynch(string $body, string $hash, string $token = null)
    {
        $headers = $this->getHeaders($token ?? $this->token);

        if ($this->env === 'test') {
            $promise = $this->client->post($this->apiUrl, [
                'body' => $body,
                'query' => $hash,
                'headers' => $headers
            ]);
        } else {
//            $client = new Client([
//                'base_uri' => $this->apiHost,
//                'connect_timeout' => $this->connectTimeout,
//                'timeout' => $this->timeout,
//            ]);
            $promise = $this->client->request('POST', $this->apiUrl, [
                'body' => $body,
                'headers' => $headers
            ]);
        }

        return $promise;

    }

    public function responseAsynch($response, $logId = null)
    {
        if ($this->client instanceof Client) {
            $result = $response->getBody()->getContents();
        }   else {
            $result = $response->getContent();
        }

        $this->logger->stop($logId ?? $this->logger->getLogId(), $result);
        $decodedBody = \json_decode($result);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage($decodedBody->data);
    }

    public function query(string $body, string $hash, string $token = null)
    {
        $logId = $this->logger->start($body);
        return $this->responseAsynch($this->queryAsynch($body, $hash, $token), $logId);
    }

    /**
     * @param array $headers
     */
    public function addHeaders(array $headers)
    {
        $this->headers = array_merge($this->headers, $headers);
    }

    /**
     * @param string $token
     *
     * @return array
     */
    public function getHeaders(string $token): array
    {
        return [
            'content-type'  => 'application/json',
            'authorization' => sprintf('Bearer %s', $token),
            // 'accept-encoding' => 'gzip, deflate'
        ];
    }

    /**
     * @param array $errors
     *
     * @return string
     */
    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }
}
