<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use GuzzleHttp\Client;
use Lexik\Bundle\JWTAuthenticationBundle\Encoder\JWTEncoderInterface;
use Symfony\Component\Cache\Adapter\AdapterInterface as Cache;

class GraphApiBridge
{
    const TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const EXPIRATION_OFFSET = 3600;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $applicationKey;

    /**
     * @var JWTEncoderInterface
     */
    private $jwtEncoder;

    /**
     * @var Cache
     */
    private $cache;

    /**
     * @var GraphApiLogger
     */
    private $logger;

    /**
     * @var GraphCoverage
     */
    private $coverage;

    /**
     * @param JWTEncoderInterface $jwtEncoder
     * @param Client              $client
     * @param Cache               $cache
     * @param GraphApiLogger      $logger
     * @param GraphCoverage       $coverage
     * @param string              $username
     * @param string              $applicationKey
     */
    public function __construct(
        JWTEncoderInterface $jwtEncoder,
        Client $client,
        Cache $cache,
        GraphApiLogger $logger,
        GraphCoverage $coverage,
        string $username,
        string $applicationKey
    ) {
        $this->jwtEncoder = $jwtEncoder;
        $this->client = $client;
        $this->cache = $cache;
        $this->logger = $logger;
        $this->coverage = $coverage;
        $this->username = $username;
        $this->applicationKey = $applicationKey;
    }

    /**
     * @param string $query
     *
     * @throws BadResponseException If the Graph API returns at least one error.
     *
     * @return Object
     */
    public function query(string $query, string $hash)
    {
        $query = $this->normalizeQuery($query);

        $tokenResponse = $this->getToken();
        $headers = $this->getHeaders($tokenResponse);

        $logId = $this->logger->start($query);
        $result = $this->client->post('/v1/', ['body' => $query, 'query' => $hash, 'headers' => $headers])
            ->getBody()
            ->getContents();
        $this->logger->stop($logId, $result);

        $decodedBody = json_decode($result);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage($decodedBody->data);
    }

    /**
     * @return string
     */
    private function getToken(): string
    {
        if ($this->cache->hasItem(self::TOKEN_CACHE_KEY)) {
            $token = $this->cache->getItem(self::TOKEN_CACHE_KEY)->get();

            if (!is_null($token) && $this->jwtEncoder->decode($token)['exp'] + self::EXPIRATION_OFFSET < time()) {
                return $token;
            }
        }

        $token = $this->jwtEncoder->encode([
            'username'        => $this->username,
            'application_key' => $this->applicationKey,
        ]);

        $item = $this->cache->getItem(self::TOKEN_CACHE_KEY);
        $item->set($token);
        $this->cache->save($item);

        return $token;
    }

    /**
     * @param string $token
     *
     * @return array
     */
    private function getHeaders(string $token): array
    {
        return [
            'content-type'  => 'application/json',
            'authorization' => sprintf('Bearer %s', $token),
        ];
    }

    /**
     * @param array $errors
     *
     * @return string
     */
    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }

    /**
     * @param string $query
     *
     * @return string
     */
    private function normalizeQuery(string $query): string
    {
        return sprintf('{"query": "%s"}', addslashes(preg_replace('!\s+!', ' ', $query)));
    }
}
