<?php

declare(strict_types=1);

namespace Tests\AlloCine\GraphClient\CoverageChecker\Checker;

use AlloCine\GraphClient\CoverageChecker\ArrayWrapper;
use AlloCine\GraphClient\CoverageChecker\Checker\RelativeChecker;
use PHPUnit\Framework\TestCase;
use stdClass;

class RelativeCheckerTest extends TestCase
{
    public function testCheck(): void
    {
        $wrapper = $this->getWrapper();
        $checker = new RelativeChecker();

        $wrapper['foobar'];

        $errors = $checker->check($wrapper);

        $this->assertCount(3, $errors);
        $this->assertSame('Array at path "foo.bar" has a too low coverage ratio (0%).', $errors[0]->getMessage());
        $this->assertSame('Array at path "foo" has a too low coverage ratio (0%).', $errors[1]->getMessage());
        $this->assertSame('Root array has a too low coverage ratio (28%).', $errors[2]->getMessage());
    }

    public function testRootCheck(): void
    {
        $wrapper = $this->getWrapper();
        $checker = new RelativeChecker();

        $wrapper['foo']['bar'][0];

        $errors = $checker->check($wrapper);
        $this->assertCount(1, $errors);
        $this->assertSame('Root array has a too low coverage ratio (71%).', $errors[0]->getMessage());
    }

    public function testThreshold(): void
    {
        $wrapper = $this->getWrapper();
        $checker = new RelativeChecker(70);

        $wrapper['foo']['bar'][0];

        $errors = $checker->check($wrapper);

        $this->assertCount(0, $errors);
    }

    public function testCheckObject(): void
    {
        $wrapper = $this->getWrapperObject();
        $checker = new RelativeChecker();

        $wrapper->foobar;

        $errors = $checker->check($wrapper);

        $this->assertCount(3, $errors);
        $this->assertSame('Array at path "foo.bar" has a too low coverage ratio (0%).', $errors[0]->getMessage());
        $this->assertSame('Array at path "foo" has a too low coverage ratio (0%).', $errors[1]->getMessage());
        $this->assertSame('Root array has a too low coverage ratio (28%).', $errors[2]->getMessage());
    }

    public function testRootCheckObject(): void
    {
        $wrapper = $this->getWrapperObject();
        $checker = new RelativeChecker();

        $wrapper->foo->bar[0];

        $errors = $checker->check($wrapper);
        $this->assertCount(1, $errors);
        $this->assertSame('Root array has a too low coverage ratio (71%).', $errors[0]->getMessage());
    }

    public function testThresholdObject(): void
    {
        $wrapper = $this->getWrapperObject();
        $checker = new RelativeChecker(70);

        $wrapper->foo->bar[0];

        $errors = $checker->check($wrapper);

        $this->assertCount(0, $errors);
    }

    private function getWrapper(): ArrayWrapper
    {
        return new ArrayWrapper([
            'foo' => [
                'bar' => ['baz']
            ],
            'foobar' => 1
        ]);
    }

    private function getWrapperObject(): ArrayWrapper
    {
        $bar = new stdClass();
        $bar->bar = ['baz'];
        $foo = new stdClass();
        $foo->foo = $bar;
        $foo->foobar = 1;

        return new ArrayWrapper($foo);
    }
}
