<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\CoverageChecker;

use ArrayAccess;
use stdClass;
use OutOfBoundsException;

class ArrayWrapper implements ArrayAccess
{
    private array $source;

    private array $usage;

    public function __construct(array|stdClass $source, private readonly string $path = '')
    {
        if ($source instanceof stdClass) {
            $source = $this->arrayCastRecursive($source);
        }

        $this->usage = array_map(static fn($item): Usage => new Usage($item), $source);

        $this->source = [];

        foreach ($source as $name => $value) {
            $subPath = $this->path ? sprintf('%s.%s', $this->path, $name) : $name;
            $this->source[$name] = is_array($value) ? new ArrayWrapper($value, $subPath) : $value;
        }
    }

    public function offsetExists($offset): bool
    {
        return array_key_exists($offset, $this->source);
    }

    public function __isset($name)
    {
        return $this->offsetExists($name);
    }

    public function offsetGet($offset): mixed
    {
        if (!$this->offsetExists($offset)) {
            throw new OutOfBoundsException(sprintf('Undefined offset "%s"', $offset));
        }

        $this->usage[$offset]->markUsed();

        return $this->source[$offset] ?? null;
    }

    public function __get(string $name)
    {
        return $this->offsetGet($name);
    }

    public function offsetSet($offset, $value): void
    {
        if (is_array($value)) {
            $value = new ArrayWrapper($value);
        }

        $this->source[$offset] = $value;
    }

    public function __set($name, $value)
    {
        $this->offsetSet($name, $value);
    }

    public function offsetUnset($offset): void
    {
        if ($this->source[$offset] ?? false) {
            unset($this->source[$offset]);
        }
    }

    public function __unset($name)
    {
        $this->offsetUnset($name);
    }

    /**
     * Iterates the source array without triggering the usage detection.
     */
    public function silkTouchIterate(callable $callback): void
    {
        foreach ($this->source as $item) {
            $callback($item);
        }
    }

    public function getPath(): string
    {
        return $this->path;
    }

    public function getCoverage(): array
    {
        $uncovered = 0;
        $covered = 0;

        foreach ($this->usage as $key => $used) {
            if ($this->source[$key] instanceof self) {
                $coverage = $this->source[$key]->getCoverage();
                $covered += $coverage['covered'];
                $uncovered += $coverage['uncovered'];
            } elseif ($used->isUsed()) {
                $covered += $used->getWeight();
            } else {
                $uncovered += $used->getWeight();
            }
        }

        return [
            'covered'   => $covered,
            'uncovered' => $uncovered,
        ];
    }

    private function arrayCastRecursive(stdClass|array $array): array
    {
        if (is_array($array)) {
            foreach ($array as $key => $value) {
                if (is_array($value)) {
                    $array[$key] = $this->arrayCastRecursive($value);
                }
                if ($value instanceof stdClass) {
                    $array[$key] = $this->arrayCastRecursive((array)$value);
                }
            }
        }

        if ($array instanceof stdClass) {
            return $this->arrayCastRecursive((array)$array);
        }

        return $array;
    }
}
