<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\FixtureFileNotFoundException;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Contracts\Cache\CacheInterface;
use Symfony\Contracts\HttpClient\ResponseInterface;

class GraphClientFixturesMiddleware
{
    private readonly CacheInterface $cache;

    public function __construct(string $apiHost, string $fixturesPath, private readonly string $mode)
    {
        $this->cache = new FilesystemAdapter($this->sanitizeRedisKey($apiHost), 0, $fixturesPath);
    }

    public function getMode(): string
    {
        return $this->mode;
    }

    public function getFixture(string $method, string $url, ?array $options): Response
    {
        $key = $this->getKey($method, $url, $options['query']);
        $fixtureItem = $this->cache->getItem($key);

        if (!$fixtureItem->isHit()) {
            throw new FixtureFileNotFoundException(sprintf('missing file : %s', $key), Response::HTTP_NOT_FOUND);
        }

        return new Response($fixtureItem->get());
    }

    private function getKey(string $method, string $url, string $hash): string
    {
        return $this->sanitizeRedisKey(sprintf('%s_%s_%s', $method, $url, $hash));
    }

    private function sanitizeRedisKey(string $key): string
    {
        return str_replace(str_split('@{}()/\:'), '_', $key);
    }

    public function setFixture(string $method, string $url, string $hash, ResponseInterface $response): bool
    {
        $fixtureItem = $this->cache->getItem($this->getKey($method, $url, $hash));
        $fixtureItem->set(($response->getContent()));

        return $this->cache->save($fixtureItem);
    }
}
