<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Client;

use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use AlloCine\GraphClient\CoverageChecker\ArrayWrapper;
use stdClass;
use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class GraphApiBridge
{
    const string TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const int EXPIRATION_OFFSET = 3600;

    private HttpClientInterface $client;

    private array $headers = [];

    public function __construct(
        HttpClientInterface $graphHttpClient,
        private readonly GraphApiLogger $logger,
        private readonly GraphCoverage $coverage,
        private readonly GraphClientFixturesMiddleware $fixturesMiddleware,
        private readonly string $env,
        private readonly string $apiUri
    ) {
        $this->client = $graphHttpClient;
    }

    /**
     * @throws BadResponseException If the Graph API returns at least one error.
     * @throws ClientExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function query(string $body, string $hash, ?string $token = null): ArrayWrapper|stdClass
    {
        $logId = $this->logger->start($body);

        if (null !== $token) {
            $this->addHeaders([
                'authorization' => sprintf('Bearer %s', $token),
            ]);
        }

        if ('test' === $this->env) {
            $mode = $this->fixturesMiddleware->getMode();
            if ($mode === 'record') {
                $response = $this->client->request('POST', $this->apiUri, [
                    'body' => $body,
                    'headers' => $this->headers,
                ]);
                $this->fixturesMiddleware->setFixture('POST', $this->apiUri, $hash, $response);
            } else {
                $response = $this->fixturesMiddleware->getFixture('POST', $this->apiUri, [
                    'query' => $hash,
                    'headers' => $this->headers,
                ]);
            }
        } else {
            $response = $this->client->request('POST', $this->apiUri, [
                'body' => $body,
                'headers' => $this->headers,
            ]);
        }

        $result = $response->getContent();

        $this->logger->stop($logId, $result);

        $decodedBody = \json_decode((string)$result, false, 512, JSON_THROW_ON_ERROR);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage((object) $decodedBody->data);
    }

    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }

    public function addHeaders(array $headers): void
    {
        $this->headers = array_merge($this->headers, $headers);
    }
}
