<?php

namespace AlloCine\GraphClient\Bundle\Client;

use Symfony\Component\HttpKernel\DataCollector\DataCollector;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class GraphCollector extends DataCollector
{
    /**
     * @var GraphApiLogger
     */
    private $logger;

    /**
     * @var GraphCoverage
     */
    private $coverage;

    /**
     * @param GraphApiLogger $logger
     * @param GraphCoverage  $coverage
     */
    public function __construct(GraphApiLogger $logger, GraphCoverage $coverage)
    {
        $this->logger = $logger;
        $this->coverage = $coverage;
    }

    /**
     * @param Request         $request
     * @param Response        $response
     * @param \Exception|null $exception
     */
    public function collect(Request $request, Response $response, \Exception $exception = null)
    {
        $this->data = [
            'queries'        => $this->logger->getAll(),
            'logger_enabled' => $this->logger->isEnabled(),
            'coverage_enabled' => $this->coverage->isEnabled(),
            'coverage_results' => $this->coverage->getResults(),
        ];
    }

    /**
     * @return array
     */
    public function getQueries()
    {
        return $this->data['queries'];
    }

    /**
     * @return int
     */
    public function getCount(): int
    {
        return count($this->getQueries());
    }

    /**
     * @return int
     */
    public function getTotalAbsolute(): int
    {
        return $this->coverage->getTotalAbsolute();
    }

    /**
     * @return int
     */
    public function getTotalRelative(): int
    {
        return $this->coverage->getTotalRelative();
    }

    /**
     * @return float
     */
    public function getTotalTime(): float
    {
        $total = array_reduce($this->getQueries(), function ($carry, $item) {
            $total = $carry;
            if (!empty($item['duration'])) {
                $total += $item['duration'];
            }

            return $total;
        });

        return round($total * 1000, 2);
    }

    /**
     * @return bool
     */
    public function isLoggerEnabled(): bool
    {
        return $this->data['logger_enabled'];
    }

    /**
     * @return bool
     */
    public function isCoverageEnabled(): bool
    {
        return $this->data['coverage_enabled'];
    }

    /**
     * @return string
     */
    public function getName()
    {
        return 'app.graph_collector';
    }

    public function reset()
    {
        $this->data = [];
    }
}
