<?php

namespace AlloCine\GraphClient;

use AlloCine\GraphClient\Bundle\Client\GraphApiBridge;
use AlloCine\GraphClient\Bundle\Client\GraphClient as BaseGraphClient;
use AlloCine\GraphClient\Bundle\Client\GraphCursor;
use AlloCine\GraphClient\Bundle\Client\BatchRequest;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use AppBundle\Cache\Traits\CachePoolTrait;
use AppBundle\Event\GraphClientEvent;
use AppBundle\Service\GraphClientInterface;
use GraphBundle\Serializer\GraphSerializer;
use GraphBundle\Serializer\Immutable\ArrayImmutable;
use GraphBundle\Serializer\Immutable\ObjectImmutable;
use Psr\Cache\CacheItemInterface;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

/**
 * This wrapper adds cache to any graph request.
 * A special treatment ensures that if the graph client fails, the request is served from the cache.
 * So, if the cache of each request contains the timestamp of the expiration and it is this one that will be tested.
 * A counter ($retry_count) has also been added to retry failed requests by spacing the attempts (see getRetryTtl()).
 * Also, the elements are cached with an infinite ttl to avoid having keys that drop out.
 */
final class OutsideGraphClient extends BaseGraphClient implements GraphClientInterface
{
    use CachePoolTrait;

    private const DEFAULT_TTL = 600;
    private const RETRY_DELAY = 30;
    private const RETRY_MULTIPLIER = 2;

    /** @var GraphSerializer */
    private $serializer;

    private $eventDispatcher;

    private $apiHost;

    public function __construct(
        GraphApiBridge $bridge,
        GraphCursor $cursor,
        QueryParser $parser,
        PhpArrayAdapter $warmUp,
        GraphSerializer $serializer,
        EventDispatcherInterface $eventDispatcher,
        string $apiHost
    ) {
        parent::__construct($bridge, $cursor, $parser, $warmUp);
        $this->serializer = $serializer;
        $this->eventDispatcher = $eventDispatcher;
        $this->apiHost = $apiHost;
    }

    /**
     * @throws \Symfony\Component\Serializer\Exception\ExceptionInterface
     *
     * @return ArrayImmutable|ObjectImmutable
     */
    public function normalize(?\stdClass $graphObject)
    {
        return $this->serializer->normalize($graphObject);
    }

    public function getResults($ttl = self::DEFAULT_TTL, bool $force = false, bool &$isFresh = false): ?\stdClass
    {
        $cacheKey = $this->getCacheKey();
        if (false === $ttl || true === $force) {
            $isFresh = true;

            return $this->injectCacheKey(parent::getResults(), $cacheKey);
        }

        $cacheItem = $this->cachePool->getItem($cacheKey);
        if ($cacheItem->isHit()) {
            $value = $cacheItem->get();
            if (time() < $value['expireAt']) {
                $isFresh = false;

                return $this->injectCacheKey($value['value'], $cacheKey);
            }

            try {
                return $this->getFreshData($cacheKey, $ttl, $cacheItem, $isFresh);
            } catch (TransportExceptionInterface $e) {
                // Connection failed, we serve old data & we extend the cache ttl
                $this->extendCache($value['value'], $cacheItem, $value['retry_count'], $ttl);
                $isFresh = false;

                return $this->injectCacheKey($value['value'], $cacheKey);
            }
        }

        return $this->getFreshData($cacheKey, $ttl, $cacheItem, $isFresh);
    }

    /**
     * @throws \AlloCine\GraphClient\Bundle\Exception\BadResponseException
     * @throws \Symfony\Component\Serializer\Exception\ExceptionInterface
     *
     * @return ArrayImmutable|ObjectImmutable
     */
    public function getNormalizedResults(int $ttl = self::DEFAULT_TTL, ?\stdClass $graphObject = null)
    {
        if (null === $graphObject) {
            $graphObject = $this->getResults($ttl);
        }

        return $this->normalize($graphObject);
    }

    /**
     * Get normalized results for multiple GraphQL queries in parallel
     *
     * @param BatchRequest $batchRequest Collection of requests to execute in parallel
     * @param callable|null $resultCallback Optional callback for streaming: function(string $key, ArrayImmutable|ObjectImmutable, Request $request, bool $fromCache): void
     *                                      If null, returns array of normalized results. If provided, calls callback for each normalized result.
     *
     * @throws \AlloCine\GraphClient\Bundle\Exception\BadResponseException
     * @throws \Symfony\Component\Serializer\Exception\ExceptionInterface
     *
     * @return array|void Array of normalized results if no callback provided, void if callback provided
     */
    public function getNormalizedMultipleResults(BatchRequest $batchRequest, ?callable $resultCallback = null)
    {
        if ($resultCallback) {
            // Streaming mode with normalization
            $this->getMultipleResults(
                $batchRequest,
                function(string $key, Result $result) use ($resultCallback) {
                    if ($result->isException()) {
                        // Pass exception as-is
                        $resultCallback($key, $result->getException(), $result->getRequest(), $result->isFromCache());
                    } else {
                        // Get the data and normalize it
                        $data = $result->getData();
                        $normalizedResult = $this->normalize($data);
                        $resultCallback($key, $normalizedResult, $result->getRequest(), $result->isFromCache());
                    }
                }
            );
        } else {
            // Batch mode with normalization
            $results = $this->getMultipleResults($batchRequest);
            $normalizedResults = [];

            foreach ($results as $key => $result) {
                if ($result->isException()) {
                    // Keep exceptions as-is
                    $normalizedResults[$key] = $result->getException();
                } else {
                    // Get the data and normalize it
                    $data = $result->getData();
                    $normalizedResults[$key] = $this->normalize($data);
                }
            }

            return $normalizedResults;
        }
    }

    /**
     * @param array $variables
     *
     * @return self
     */
    public function setVariables(array $variables): BaseGraphClient
    {
        $event = new GraphClientEvent($variables);
        $this->eventDispatcher->dispatch($event, GraphClientEvent::PRE_SET_GRAPH_CLIENT_VARIABLES);

        return parent::setVariables($event->getVariables());
    }

    private function getCacheKey(): string
    {
        return \sprintf('%s_%s_v2', \md5($this->apiHost), $this->computeHash());
    }

    private function getFreshData(string $cacheKey, int $ttl, CacheItemInterface $cacheItem, bool &$isFresh): ?\stdClass
    {
        $results = parent::getResults();
        $packedResults = $this->packValueWithMetadata($results, $ttl);
        $isFresh = true;
        $cacheItem->set($packedResults);
        $this->cachePool->save($cacheItem);

        return $this->injectCacheKey($results, $cacheKey);
    }

    private function injectCacheKey(?\stdClass $graphObject, string $cacheKey): ?\stdClass
    {
        if (is_object($graphObject)) {
            $graphObject->cacheKey = $cacheKey;
        }

        return $graphObject;
    }

    private function packValueWithMetadata($value, int $ttl, int $retryCount = 0): array
    {
        return [
            'expireAt' => time() + $ttl,
            'value' => $value,
            'retry_count' => $retryCount,
        ];
    }

    /**
     * Calculate the retry ttl.
     *
     * Calculation is :
     *
     * delay * multiplier ^ retry_count
     *
     * So with delay of 30s, multiplier of 2 :
     *
     * retry 1 : 30 * (2 ^ 1) = 60s
     * retry 2 : 30 * (2 ^ 2) = 120s
     * retry 3 : 30 * (2 ^ 3) = 240s
     *
     * A max_delay with initial ttl value has been added.
     */
    private function getRetryTtl(int $ttl, int $retryCount = 1): int
    {
        $nextTtl = self::RETRY_DELAY * (self::RETRY_MULTIPLIER ^ $retryCount);

        if ($nextTtl > $ttl) {
            return $ttl;
        }

        return $nextTtl;
    }

    private function extendCache($value, CacheItemInterface $cacheItem, int $retryCount, int $ttl): void
    {
        $cacheItem->set(
            $this->packValueWithMetadata(
                $value,
                $this->getRetryTtl($ttl, $retryCount),
                $retryCount + 1
            )
        );
        $this->cachePool->save($cacheItem);
    }

    /**
     * Execute multiple GraphQL queries in parallel with caching support
     *
     * @param BatchRequest $batchRequest Collection of requests to execute in parallel
     * @param callable|null $resultCallback Optional callback for streaming: function(string $key, $result, Request $request, bool $fromCache): void
     *                                      If null, returns array of all results. If provided, calls callback for each result.
     *
     * @return array|void Array of results if no callback provided, void if callback provided
     */
    public function getMultipleResults(BatchRequest $batchRequest, ?callable $resultCallback = null)
    {
        if ($batchRequest->isEmpty()) {
            return $resultCallback ? null : [];
        }

        // Optimized: check cache in batch to reduce overhead
        $cachableRequests = [];
        $nonCachableRequests = [];
        $cacheKeys = [];
        $cachedResults = [];

        // Separate cachable from non-cachable requests and build key mapping
        foreach ($batchRequest->getRequests() as $index => $request) {
            $resultKey = $request->getIdentifier() ?? $index; // Use identifier or fallback to index

            if ($request->getTtl() > 0) {
                $cacheKey = $this->generateBatchCacheKey($request);
                $cacheKeys[$resultKey] = $cacheKey;
                $cachableRequests[$resultKey] = $request;
            } else {
                $nonCachableRequests[$resultKey] = $request;
            }
        }

        // Batch cache lookup for better performance
        if (!empty($cachableRequests)) {
            try {
                // Build a mapping between cache keys and request keys
                $cacheKeyToRequestKey = [];
                foreach ($cacheKeys as $requestKey => $cacheKey) {
                    $cacheKeyToRequestKey[$cacheKey] = $requestKey;
                }

                // Get all cache items at once - use array conversion to avoid generator issues
                $cacheItems = iterator_to_array(
                    $this->cachePool->getItems(array_values($cacheKeys)),
                    true // preserve keys
                );

                foreach ($cacheItems as $cacheKey => $cacheItem) {
                    if (isset($cacheKeyToRequestKey[$cacheKey]) && $cacheItem->isHit()) {
                        $cachedData = $cacheItem->get();
                        if (time() < $cachedData['expireAt']) {
                            $resultKey = $cacheKeyToRequestKey[$cacheKey];
                            $request = $cachableRequests[$resultKey];
                            $cachedResult = $this->injectCacheKey($cachedData['value'], $cacheKey);

                            if ($resultCallback) {
                                $resultCallback($resultKey, $cachedResult, $request, true);
                            } else {
                                $cachedResults[$resultKey] = $cachedResult;
                            }
                            unset($cachableRequests[$resultKey]); // Remove from execution list
                        }
                    }
                }
            } catch (\Exception $e) {
                // If batch cache lookup fails, fall back to individual lookups
                foreach ($cachableRequests as $resultKey => $request) {
                    $cacheKey = $cacheKeys[$resultKey];
                    $cacheItem = $this->cachePool->getItem($cacheKey);
                    if ($cacheItem->isHit()) {
                        $cachedData = $cacheItem->get();
                        if (time() < $cachedData['expireAt']) {
                            $cachedResult = $this->injectCacheKey($cachedData['value'], $cacheKey);

                            if ($resultCallback) {
                                $resultCallback($resultKey, $cachedResult, $request, true);
                            } else {
                                $cachedResults[$resultKey] = $cachedResult;
                            }
                            unset($cachableRequests[$resultKey]);
                        }
                    }
                }
            }
        }

        // Combine remaining cachable requests with non-cachable ones
        $requestsToExecute = $cachableRequests + $nonCachableRequests;

        if (empty($requestsToExecute)) {
            return $resultCallback ? null : $cachedResults;
        }

        // Dispatch batch event for requests to execute
        $this->dispatchBatchEventOptimized($requestsToExecute);

        // Execute remaining requests
        if ($resultCallback) {
            $this->executeOptimizedStreamingMode($requestsToExecute, $cacheKeys, $resultCallback);
        } else {
            return $this->executeOptimizedBatchMode($requestsToExecute, $cacheKeys, $cachedResults, $batchRequest);
        }
    }

    /**
     * Optimized dispatch event for batch execution
     */
    private function dispatchBatchEventOptimized(array $requestsToExecute): void
    {
        if ($this->eventDispatcher) {
            $batchVariables = [];
            foreach ($requestsToExecute as $index => $request) {
                $batchVariables[$index] = $request->getVariables();
            }

            $event = new GraphClientEvent($batchVariables, true);
            $this->eventDispatcher->dispatch($event, GraphClientEvent::PRE_SET_GRAPH_CLIENT_VARIABLES);

            // Apply modified variables back to requests
            $modifiedVariables = $event->getVariables();
            foreach ($requestsToExecute as $index => $request) {
                if (isset($modifiedVariables[$index])) {
                    $request->setVariables($modifiedVariables[$index]);
                }
            }
        }
    }

    /**
     * Optimized streaming mode execution
     */
    private function executeOptimizedStreamingMode(array $requestsToExecute, array $cacheKeys, callable $resultCallback): void
    {
        // Create batch request for parent call
        $executeBatch = new BatchRequest();
        foreach ($requestsToExecute as $request) {
            $executeBatch->addRequest($request);
        }

        $indexMapping = array_keys($requestsToExecute);

        parent::getMultipleResults(
            $executeBatch,
            function(int $executeIndex, $result, $request) use ($resultCallback, $indexMapping, $cacheKeys, $requestsToExecute) {
                $originalIndex = $indexMapping[$executeIndex];
                $originalRequest = $requestsToExecute[$originalIndex];

                // Cache successful results
                if ($originalRequest->getTtl() > 0 && !($result instanceof \Exception) && isset($cacheKeys[$originalIndex])) {
                    $this->cacheBatchResult($cacheKeys[$originalIndex], $result, $originalRequest->getTtl());
                }

                $resultCallback($originalIndex, $result, $request, false);
            }
        );
    }

    /**
     * Optimized batch mode execution
     */
    private function executeOptimizedBatchMode(array $requestsToExecute, array $cacheKeys, array $cachedResults, BatchRequest $originalBatch): array
    {
        try {
            // Create batch request for parent call
            $executeBatch = new BatchRequest();
            foreach ($requestsToExecute as $request) {
                $executeBatch->addRequest($request);
            }

            $executedResults = parent::getMultipleResults($executeBatch);

            // Merge results
            $finalResults = $cachedResults;
            foreach ($executedResults as $executeIndex => $result) {
                $originalRequest = $requestsToExecute[$executeIndex];
                $finalResults[$executeIndex] = $result;

                // Cache successful results
                if ($originalRequest->getTtl() > 0 && !($result instanceof \Exception) && isset($cacheKeys[$executeIndex])) {
                    $this->cacheBatchResult($cacheKeys[$executeIndex], $result, $originalRequest->getTtl());
                }
            }

            ksort($finalResults);
            return $finalResults;

        } catch (\Exception $e) {
            return $this->handleBatchError($originalBatch, $cacheKeys, $cachedResults, $e);
        }
    }

    /**
     * Generate cache key for a single request in a batch
     */
    private function generateBatchCacheKey($request): string
    {
        $keyData = [
            'query' => $request->getQueryName(),
            'variables' => $request->getVariables(),
            'host' => $this->apiHost
        ];

        return sprintf('%s_batch_%s_v2', md5($this->apiHost), md5(serialize($keyData)));
    }

    /**
     * Cache a batch result with expiration using the same pattern as existing cache
     */
    private function cacheBatchResult(string $cacheKey, Result $result, int $ttl): void
    {
        $cacheItem = $this->cachePool->getItem($cacheKey);
        $packedResult = $this->packValueWithMetadata($result->getData(), $ttl);
        $cacheItem->set($packedResult);
        $this->cachePool->save($cacheItem);
    }

    /**
     * Handle batch execution errors by serving from cache when possible
     */
    private function handleBatchError(BatchRequest $batchRequest, array $cacheKeys, array $cachedResults, \Exception $error): array
    {
        $results = $cachedResults;

        // For requests not in cache, try to serve stale cache or return error
        foreach ($batchRequest->getRequests() as $index => $request) {
            if (!isset($results[$index]) && isset($cacheKeys[$index])) {
                $cacheItem = $this->cachePool->getItem($cacheKeys[$index]);
                if ($cacheItem->isHit()) {
                    // Serve stale cache in case of error and extend cache
                    $cachedData = $cacheItem->get();
                    $results[$index] = $this->injectCacheKey($cachedData['value'], $cacheKeys[$index]);

                    // Extend cache with retry logic using request's individual TTL
                    $ttl = $request->getTtl();
                    $this->extendCache($cachedData['value'], $cacheItem, $cachedData['retry_count'] ?? 0, $ttl);
                } else {
                    // No cache available, re-throw the error
                    throw $error;
                }
            }
        }

        ksort($results);
        return $results;
    }
}
