<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Client;

use Throwable;
use Override;
use Exception;
use Symfony\Component\HttpKernel\DataCollector\DataCollector;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class GraphCollector extends DataCollector
{
    public function __construct(private readonly GraphApiLogger $logger, private readonly GraphCoverage $coverage)
    {
    }

    public function collect(Request $request, Response $response, ?Throwable $exception = null): void
    {
        $this->data = [
            'queries' => $this->logger->getAll(),
            'logger_enabled' => $this->logger->isEnabled(),
            'coverage_enabled' => $this->coverage->isEnabled(),
            'coverage_results' => $this->coverage->getResults(),
        ];
    }

    public function getQueries(): array
    {
        return $this->data['queries'];
    }

    public function getCount(): int
    {
        return count($this->getQueries());
    }

    public function getTotalAbsolute(): int
    {
        return $this->coverage->getTotalAbsolute();
    }

    public function getTotalRelative(): int
    {
        return $this->coverage->getTotalRelative();
    }

    public function getTotalTime(): float
    {
        $queries = $this->getQueries();

        if (empty($queries)) {
            return 0;
        }

        // For batch processing, calculate actual wall-clock time
        // from earliest start to latest end time
        $startTimes = [];
        $endTimes = [];

        foreach ($queries as $query) {
            if (!empty($query['start_time'])) {
                $startTimes[] = $query['start_time'];
            }
            if (!empty($query['end_time'])) {
                $endTimes[] = $query['end_time'];
            }
        }

        if (empty($startTimes) || empty($endTimes)) {
            // Fallback to sum of durations if timing data is incomplete
            $total = array_reduce($queries, function ($carry, $item) {
                return $carry + ($item['duration'] ?? 0);
            }, 0);
            return round($total * 1000, 2);
        }

        // Calculate actual wall-clock time for parallel execution
        $earliestStart = min($startTimes);
        $latestEnd = max($endTimes);
        $wallClockTime = $latestEnd - $earliestStart;

        return round($wallClockTime * 1000, 2);
    }

    public function getMaxComplexity(): float
    {
        if (count($this->getQueries()) > 0 && isset($this->getQueries()[0]['complexity'])) {
            return max(array_column($this->getQueries(), 'complexity'));
        }

        return 0;
    }

    public function isLoggerEnabled(): bool
    {
        return $this->data['logger_enabled'];
    }

    public function isCoverageEnabled(): bool
    {
        return $this->data['coverage_enabled'];
    }

    public function getName(): string
    {
        return 'app.graph_collector';
    }

    public function reset()
    {
        $this->data = [];
    }

    /**
     * @return int
     */
    public function getBatchedCount(): int
    {
        return count(array_filter($this->getQueries(), function($query) {
            return $query['is_batch'] ?? false;
        }));
    }

    /**
     * @return int
     */
    public function getNonBatchedCount(): int
    {
        return count(array_filter($this->getQueries(), function($query) {
            return !($query['is_batch'] ?? false);
        }));
    }

    /**
     * @return float
     */
    public function getBatchedTime(): float
    {
        $batchedQueries = array_filter($this->getQueries(), function($query) {
            return $query['is_batch'] ?? false;
        });

        if (empty($batchedQueries)) {
            return 0;
        }

        // For batched queries, calculate wall-clock time
        $startTimes = [];
        $endTimes = [];

        foreach ($batchedQueries as $query) {
            if (!empty($query['start_time'])) {
                $startTimes[] = $query['start_time'];
            }
            if (!empty($query['end_time'])) {
                $endTimes[] = $query['end_time'];
            }
        }

        if (empty($startTimes) || empty($endTimes)) {
            return 0;
        }

        $earliestStart = min($startTimes);
        $latestEnd = max($endTimes);
        $wallClockTime = $latestEnd - $earliestStart;

        return round($wallClockTime * 1000, 2);
    }

    /**
     * @return float
     */
    public function getNonBatchedTime(): float
    {
        $nonBatchedQueries = array_filter($this->getQueries(), function($query) {
            return !($query['is_batch'] ?? false);
        });

        $total = array_reduce($nonBatchedQueries, function ($carry, $item) {
            return $carry + ($item['duration'] ?? 0);
        }, 0);

        return round($total * 1000, 2);
    }

    /**
     * @return bool
     */
    public function hasBatchedQueries(): bool
    {
        return $this->getBatchedCount() > 0;
    }

    /**
     * @return bool
     */
    public function hasNonBatchedQueries(): bool
    {
        return $this->getNonBatchedCount() > 0;
    }
}
