<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use AlloCine\GraphClient\Bundle\Exception\FileNotFoundException;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use stdClass;

class GraphClient
{
    private ?string $query = null;

    private ?string $queryName = null;

    private ?string $preparedQuery;

    private array $variables = [];

    private ?string $uri = null;

    public function __construct(
        protected GraphApiBridge $bridge,
        private readonly GraphCursor $cursor,
        private readonly QueryParser $parser,
        private readonly PhpArrayAdapter $warmUp
    ) {
    }

    protected function computeHash(): string
    {
        return \md5(\sprintf('%s_%s', $this->queryName, \serialize($this->variables)));
    }

    /**
     * Compute hash for testing purpose (building and finding fixtures)
     */
    protected function computeTestHash(): string
    {
        $variables = $this->variables;
        \array_walk_recursive(
            $variables,
            static function (&$value): void {
                if (\is_string($value) && 0 < \preg_match('#^(\d{4}-)?(0[1-9]|1[0-2])-(0[1-9]|[1-2]\d|3[0-1])$#', $value)) {
                    $value = null;
                }
            }
        );

        $query = \preg_replace('#(\d{4}-)?(0[1-9]|1[0-2])-(0[1-9]|[1-2]\d|3[0-1])#', '', $this->getQuery());

        return \md5(\sprintf('%s_%s_%s', $this->uri, $query, \serialize($variables)));
    }

    public function getQuery(): string
    {
        return $this->query ?? $this->preparedQuery;
    }

    public function setUri(string $uri): GraphClient
    {
        $this->uri = $uri;

        return $this;
    }

    public function setVariables(array $variables): GraphClient
    {
        $this->variables = $variables;

        return $this;
    }

    public function getAfterCursor(int $cursor, int $offset = 0): ?string
    {
        return $this->cursor->getAfterCursor($cursor, $offset);
    }

    /**
     * @throws InvalidArgumentException
     */
    public function prepare(string $query): GraphClient
    {
        $this->preparedQuery = $this->parser
            ->setFragments(
                $this->warmUp
                    ->getItem(GraphApiCacheWarmer::CACHED_FRAGMENT_KEY)
                    ->get()
            )
            ->parseQuery($query);

        return $this;
    }

    /**
     * @throws InvalidArgumentException
     */
    public function generateQuery(?string $queryName = null): GraphClient
    {
        $this->queryName = $queryName;
        $query = $queryName ? $this->warmUp->getItem($queryName)->get() : null;
        if (null === $this->preparedQuery && null === $query) {
            throw new FileNotFoundException("The graph query file $queryName does not exist");
        }

        if (null !== $this->preparedQuery) {
            $query = $this->preparedQuery;
        }
        $this->parseQuery($query);

        return $this;
    }

    /**
     * @throws BadResponseException
     */
    public function getResults(): ?stdClass
    {
        $result = $this->bridge->query($this->query, $this->computeTestHash());
        $this->init();

        return $result;
    }

    /**
     * Execute multiple GraphQL queries in parallel using a BatchRequest
     *
     * @param BatchRequest $batchRequest Collection of requests to execute in parallel
     * @param callable|null $resultCallback Optional callback for streaming: function(int $index, Result $result): void
     *                                      If null, returns array of Result objects. If provided, calls callback for each result.
     *
     * @throws BadResponseException
     * @throws \Psr\Cache\InvalidArgumentException
     *
     * @return array|void Array of Result objects if no callback provided, void if callback provided
     */
    public function getMultipleResults(BatchRequest $batchRequest, ?callable $resultCallback = null)
    {
        if ($batchRequest->isEmpty()) {
            return $resultCallback ? null : [];
        }

        $queries = [];
        $requestsMap = [];

        // Prepare all queries
        foreach ($batchRequest->getRequests() as $index => $request) {
            // Set up the query temporarily
            $this->variables = $request->getVariables();
            $this->queryName = $request->getQueryName();

            // Generate the query
            $query = $this->warmUp->getItem($request->getQueryName())->get();
            if (null === $query) {
                throw new FileNotFoundException("The graph query file {$request->getQueryName()} does not exist");
            }

            $this->parseQuery($query);

            // Prepare query data for parallel execution
            $queries[$index] = [
                'body' => $this->query,
                'hash' => $this->computeTestHashForQuery($request->getQueryName(), $request->getVariables())
            ];

            $requestsMap[$index] = $request;
        }

        // Clean up early
        $this->init();

        // Execute all prepared queries in parallel
        if ($resultCallback) {
            // Streaming mode - create Result objects and pass to callback
            $this->bridge->queryMultiple($queries, function(int $index, $rawResult) use ($resultCallback, $requestsMap) {
                $request = $requestsMap[$index];
                $resultKey = $request->getIdentifier() ?? $index;
                $result = new Result($rawResult, $request, false, $resultKey);
                $resultCallback($index, $result);
            });
        } else {
            // Batch mode - return array of Result objects
            $rawResults = $this->bridge->queryMultiple($queries);
            $finalResults = [];

            foreach ($rawResults as $resultIndex => $rawResult) {
                if (isset($requestsMap[$resultIndex])) {
                    $request = $requestsMap[$resultIndex];
                    $resultKey = $request->getIdentifier() ?? $resultIndex;
                    $result = new Result($rawResult, $request, false, $resultKey);
                    $finalResults[$resultKey] = $result;
                }
            }

            return $finalResults;
        }
    }

    /**
     * Compute test hash for a specific query and variables combination
     *
     * @param string $queryName
     * @param array $variables
     * @return string
     */
    private function computeTestHashForQuery(string $queryName, array $variables): string
    {
        $cleanedVariables = $variables;
        \array_walk_recursive(
            $cleanedVariables,
            function (&$value, $key) {
                if (\is_string($value) && 0 < \preg_match('#^([0-9]{4}-)?(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])$#', $value)) {
                    $value = null;
                }
            }
        );

        $query = $this->warmUp->getItem($queryName)->get();
        $cleanedQuery = \preg_replace('#([0-9]{4}-)?(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])#', '', $query);

        return \md5(\sprintf('%s_%s_%s', $this->uri, $cleanedQuery, \serialize($cleanedVariables)));
    }

    /**
     * @param string $query
     */
    private function parseQuery(string $query): void
    {
        $variables = $this->variables === [] ? '' : sprintf(', "variables": %s',
            \json_encode($this->variables, JSON_THROW_ON_ERROR)
        );

        $this->query = \sprintf(
            '{"query": "%s"%s}',
            addslashes((string) preg_replace('@\s+@', ' ', $query)),
            $variables
        );
    }

    private function init(): void
    {
        $this->query = null;
        $this->preparedQuery = null;
        $this->uri = null;
        $this->variables = [];
        $this->queryName = null;
    }
}
