<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use GuzzleHttp\Client;
use Symfony\Component\Cache\Adapter\AdapterInterface as Cache;

class GraphApiBridge
{
    const TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const EXPIRATION_OFFSET = 3600;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $token;

    /**
     * @var string
     */
    private $apiHost;

    /**
     * @var string
     */
    private $apiUri;

    /**
     * @var string
     */
    private $apiCountryName;

    /**
     * @var array
     */
    private $headers;

    /**
     * @var string
     */
    private $env;

    /**
     * @var Cache
     */
    private $cache;

    /**
     * @var GraphApiLogger
     */
    private $logger;

    /**
     * @var GraphCoverage
     */
    private $coverage;

    /**
     * @return string
     */
    public function getApiHost(): string
    {
        return $this->apiHost;
    }

    /**
     * @return string
     */
    public function getApiUri(): string
    {
        return $this->apiUri;
    }

    /**
     * @return string
     */
    public function getApiCountryName(): string
    {
        return $this->apiCountryName;
    }

    /**
     * @param Client         $client
     * @param Cache          $cache
     * @param GraphApiLogger $logger
     * @param GraphCoverage  $coverage
     * @param string         $env
     * @param string         $token
     * @param string         $apiHost
     * @param string         $apiUri
     * @param array          $headers
     */
    public function __construct(
        Client $client,
        Cache $cache,
        GraphApiLogger $logger,
        GraphCoverage $coverage,
        string $env,
        string $token,
        string $apiHost,
        string $apiUri,
        string $apiCountryName,
        array $headers = []
    ) {
        $this->client = $client;
        $this->cache = $cache;
        $this->logger = $logger;
        $this->coverage = $coverage;
        $this->env = $env;
        $this->token = $token;
        $this->apiHost = $apiHost;
        $this->apiUri = $apiUri;
        $this->apiCountryName = $apiCountryName;
        $this->headers = $headers;
    }

    /**
     * @param string $body
     * @param string $hash
     * @param string $token
     *
     * @throws BadResponseException If the Graph API returns at least one error.
     *
     * @return Object
     */
    public function query(string $body, string $hash, string $token = null)
    {
        $headers = $this->getHeaders($token ?? $this->token);

        $logId = $this->logger->start($body);
        if ($this->env === 'test') {
            $response = $this->client->post($this->apiUri, [
                'body' => $body,
                'query' => $hash,
                'headers' => $headers
            ]);
        } else {
            $client = new Client(['base_uri' => $this->apiHost]);
            $response = $client->request('POST', $this->apiUri, [
                'body' => $body,
                'headers' => $headers
            ]);
        }

        $result = $response ->getBody()->getContents();
        $this->logger->stop($logId, $result);

        $decodedBody = json_decode($result);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage($decodedBody->data);
    }

    /**
     * @param array $headers
     */
    public function addHeaders(array $headers)
    {
        $this->headers = array_merge($this->headers, $headers);
    }

    /**
     * @param string $token
     *
     * @return array
     */
    public function getHeaders(string $token): array
    {
        $configHeaders = $this->client->getConfig('headers') ?? [];

        return array_merge($configHeaders, $this->headers, [
            'content-type'  => 'application/json',
            'authorization' => sprintf('Bearer %s', $token),
        ]);
    }

    /**
     * @param array $errors
     *
     * @return string
     */
    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }
}
