<?php

namespace AlloCine\GraphClient\Bundle\Client;

use GraphQL\Language\AST\FragmentSpreadNode;
use GraphQL\Language\AST\NodeKind;
use GraphQL\Language\Parser;
use GraphQL\Language\Visitor;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use Symfony\Component\Cache\Adapter\AbstractAdapter;

class GraphClient
{
    /**
     * @var GraphApiBridge
     */
    private $bridge;

    /**
     * @var GraphCursor
     */
    private $cursor;

    /**
     * @var AbstractAdapter
     */
    private $cache;

    /**
     * @var PhpArrayAdapter
     */
    private $warmup;

    /**
     * @var string
     */
    private $preparedQuery;

    /**
     * @var string
     */
    private $query;

    /**
     * @var array
     */
    private $loadedFragments = [];

    /**
     * GraphClient constructor.
     *
     * @param GraphApiBridge  $bridge
     * @param GraphCursor     $cursor
     * @param AbstractAdapter $cache
     * @param PhpArrayAdapter $warmup
     */
    public function __construct(
        GraphApiBridge $bridge,
        GraphCursor $cursor,
        AbstractAdapter $cache,
        PhpArrayAdapter $warmup
    ) {
        $this->bridge = $bridge;
        $this->cursor = $cursor;
        $this->cache = $cache;
        $this->warmup = $warmup;
    }

    /**
     * @param string $source
     * @param array  $criterion
     *
     * @return array
     */
    private function getFragmentNames(string $source, array $criterion): array
    {
        $fragments = [];

        $source = $this->replaceCriterion($source, $criterion);
        Visitor::visit(Parser::parse($source), [
            'leave' => [
                NodeKind::FRAGMENT_SPREAD => function (FragmentSpreadNode $node) use (&$fragments) {
                    $fragments[] = $node->name->value;
                },
            ],
        ]);

        return $fragments;
    }

    /**
     * @param string $query
     * @param array  $criterion
     *
     * @return string
     * @throws \FileNotFoundException
     */
    private function addFragments(string $query, array $criterion): string
    {
        if ($matches = $this->getFragmentNames($query, $criterion)) {
            foreach ($matches as $fragment) {
                if (!in_array($fragment, $this->loadedFragments)) {
                    $fragmentFile = $this->warmup->getItem($fragment)->get();
                    if (is_null($fragmentFile)) {
                        throw new \FileNotFoundException("The graph fragment $fragment does not exist");
                    }

                    $this->loadedFragments[] = $fragment;
                    $fragment = $this->addFragments($fragmentFile, $criterion);
                    $query .= $fragment;
                }
            }
        }

        return $query;
    }

    /**
     * @param string $query
     * @param array  $criterion
     *
     * @return string
     * @throws \Exception
     */
    private function replaceCriterion(string $query, array $criterion): string
    {
        foreach ($criterion as $replacement => $values) {
            $valueResult = [];
            if (!is_array($values)) {
                throw new \Exception('The criterion values must be an array');
            }
            foreach ($values as $key => $value) {
                $valueResult[] = "$key: $value";
            }
            $valueResult = implode(', ', $valueResult);

            $query = str_replace("%$replacement%", $valueResult, $query);
        }

        if (preg_match('@%(.*?)%@', $query) !== 0) {
            throw new \Exception('A criterion in graph query wasn\'t replaced');
        }

        return $query;
    }

    /**
     * @param $key
     *
     * @return string|null
     */
    private function getCachedQuery(string $key)
    {
        if ($this->hasCachedQuery($key)) {
            $encoded = base64_encode($key);

            return $this->cache->getItem($encoded)->get();
        }

        return null;
    }

    /**
     * @param string $key
     *
     * @return bool
     */
    private function hasCachedQuery(string $key): bool
    {
        $encoded = base64_encode($key);

        return $this->cache->hasItem($encoded);
    }

    /**
     * @param string $queryName
     * @param string $query
     */
    private function setCachedQuery(string $queryName, string $query)
    {
        if (!$this->hasCachedQuery($queryName)) {
            $encoded = base64_encode($queryName);
            $item = $this->cache->getItem($encoded)
                ->set($query);
            $this->cache->save($item);
        }
    }

    /**
     * @param string $queryName
     * @param string $query
     * @param array  $criterion
     *
     * @return string
     */
    private function addFragmentAndCacheQuery(string $queryName, string $query, array $criterion): string
    {
        $query = $this->addFragments($query, $criterion);
        $this->loadedFragments = [];
        $this->setCachedQuery($queryName, $query);

        return $query;
    }

    /**
     * @param string $preparedQuery
     *
     * @return self
     */
    public function prepare(string $preparedQuery): GraphClient
    {
        Parser::parse($preparedQuery);
        $this->preparedQuery = $preparedQuery;

        return $this;
    }

    /**
     * @param string|null $queryName
     * @param array       $criterion
     *
     * @return self
     * @throws \FileNotFoundException|\Exception
     */
    public function generateQuery(string $queryName, array $criterion = []): GraphClient
    {
        $queryFile = $this->warmup->getItem($queryName);
        if (is_null($this->preparedQuery) && is_null($queryFile)) {
            throw new \FileNotFoundException("The graph query file $queryName does not exist");
        } elseif ($this->hasCachedQuery($queryName)) {
            $query = $this->getCachedQuery($queryName);
        } elseif (is_null($this->preparedQuery)) {
            $query = $this->addFragmentAndCacheQuery($queryName, $queryFile->get(), $criterion);
        } else {
            $query = $this->addFragmentAndCacheQuery($queryName, $this->preparedQuery, $criterion);
            $this->preparedQuery = null;
        }

        $this->query = $this->replaceCriterion($query, $criterion);

        return $this;
    }

    /**
     * @return \stdClass
     */
    public function getResults(): \stdClass
    {
        return $this->bridge->query($this->query);
    }

    /**
     * @return string
     */
    public function getQuery(): string
    {
        return $this->query;
    }

    /**
     * @param int $page
     * @param int $offset
     *
     * @return null|string
     */
    public function getAfterCursor(int $page, int $offset)
    {
        return $this->cursor->getAfterCursor($page, $offset);
    }
}
