<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Tests\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\FileNotFoundException;
use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Client\GraphApiBridge;
use AlloCine\GraphClient\Bundle\Client\GraphClient;
use AlloCine\GraphClient\Bundle\Client\GraphCursor;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Exception;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class GraphClientTest extends TestCase
{
    /**
     * @var GraphApiBridge
     */
    private $bridge;

    private ?GraphCursor $cursor = null;

    private ?PhpArrayAdapter $cacheAdapter = null;

    private ?FilesystemAdapter $fileSystemAdapter = null;

    /**
     * @throws Exception|FileNotFoundException
     */
    public function setUp(): void
    {
        $this->setCacheFragments();

        $this->bridge = $this->getMockBuilder(GraphApiBridge::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->cursor = new GraphCursor();
    }

    public function testCursorValues(): void
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $this->assertNull($service->getAfterCursor(1, 0));
        $this->assertSame(base64_encode('arrayconnection:9'), $service->getAfterCursor(10));
        $this->assertSame(base64_encode('arrayconnection:29'), $service->getAfterCursor(30));
        $this->assertSame(base64_encode('arrayconnection:19'), $service->getAfterCursor(20));
        $this->assertSame(base64_encode('arrayconnection:22'), $service->getAfterCursor(20, 3));
    }

    public function testPrepareError(): void
    {
        $this->expectException(FileNotFoundException::class);
        $this->expectExceptionMessage('The graph fragment errorFragment does not exist');

        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testWithError {
                ...errorFragment
            }
        }
GRAPHQL;
        $service->prepare($query);
        $this->assertSame($query, $service->getQuery());
    }

    /**
     * @throws Exception|FileNotFoundException
     */
    public function testPrepareWithoutFragment(): void
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query);
        $this->assertSame($query, $service->getQuery());
    }

    /**
     * @throws Exception|FileNotFoundException
     */
    public function testPrepareWithFragment(): void
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                ...realFragment
            }
        }
GRAPHQL;
        $expected =
            <<<GRAPHQL
        {
            testReal {
                ...realFragment
            }
        } fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL;
        $service->prepare($query);
        $this->assertSame(rtrim($expected), rtrim($service->getQuery()));
    }

    /**
     * @throws Exception|FileNotFoundException
     */
    public function testGenerateQuery(): void
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query)->generateQuery();
        $this->assertSame('{"query": " { testReal { test } }"}', $service->getQuery());
    }

    /**
     * @throws Exception|FileNotFoundException
     */
    public function testGenerateQueryWithVariables(): void
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query)->setVariables(['order' => 'ORDER', 'first' => 4, 'bool' => true, 'facets' => [["field" => "ELEMENTS"]]])->generateQuery();
        $this->assertSame('{"query": " { testReal { test } }", "variables": {"order":"ORDER","first":4,"bool":true,"facets":[{"field":"ELEMENTS"}]}}', $service->getQuery());
    }

    /**
     * @throws Exception|FileNotFoundException
     */
    private function setCacheFragments(): void
    {
        $cacheDir = __DIR__ . '/../Resources/cache/queries.php';
        $this->fileSystemAdapter = new FilesystemAdapter('', 0, __DIR__ . '/../Resources/cache/filesystem');
        $this->cacheAdapter = new PhpArrayAdapter($cacheDir, $this->fileSystemAdapter);
        $sources = ['queries' => [__DIR__ . '/../Resources/graph/queries'], 'fragments' => [__DIR__ . '/../Resources/graph/fragments']];
        $service = new GraphApiCacheWarmer($this->cacheAdapter, new QueryParser(), $sources, '.graphql');
        $service->warmUp('');
    }
}
