<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Client;

use Throwable;
use Override;
use Exception;
use Symfony\Component\HttpKernel\DataCollector\DataCollector;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class GraphCollector extends DataCollector
{
    public function __construct(private readonly GraphApiLogger $logger, private readonly GraphCoverage $coverage)
    {
    }

    public function collect(Request $request, Response $response, ?Throwable $exception = null): void
    {
        $this->data = [
            'queries' => $this->logger->getAll(),
            'logger_enabled' => $this->logger->isEnabled(),
            'coverage_enabled' => $this->coverage->isEnabled(),
            'coverage_results' => $this->coverage->getResults(),
        ];
    }

    public function getQueries(): array
    {
        return $this->data['queries'];
    }

    public function getCount(): int
    {
        return count($this->getQueries());
    }

    public function getTotalAbsolute(): int
    {
        return $this->coverage->getTotalAbsolute();
    }

    public function getTotalRelative(): int
    {
        return $this->coverage->getTotalRelative();
    }

    public function getTotalTime(): float
    {
        $total = array_reduce($this->getQueries(), static function ($carry, $item) {
            $total = $carry;
            if (!empty($item['duration'])) {
                $total += $item['duration'];
            }

            return $total;
        });

        return round($total * 1000, 2);
    }

    public function getMaxComplexity(): float
    {
        if (count($this->getQueries()) > 0 && isset($this->getQueries()[0]['complexity'])) {
            return max(array_column($this->getQueries(), 'complexity'));
        }

        return 0;
    }

    public function isLoggerEnabled(): bool
    {
        return $this->data['logger_enabled'];
    }

    public function isCoverageEnabled(): bool
    {
        return $this->data['coverage_enabled'];
    }

    public function getName(): string
    {
        return 'app.graph_collector';
    }
}
