<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Cache;

use Exception;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\HttpKernel\CacheWarmer\CacheWarmerInterface;
use AlloCine\GraphClient\Bundle\Exception\FileNotFoundException;

class GraphApiCacheWarmer implements CacheWarmerInterface
{
    const CACHED_FRAGMENT_KEY = 'cachedFragments';

    private array $queries = [];

    private array $fragments = [];

    public function __construct(
        private readonly PhpArrayAdapter $cache,
        private readonly QueryParser $parser,
        private array $sources,
        private readonly string $extension
    ) {
        $this->init();
    }

    private function init(): void
    {
        $this->queries = $this->storedQueries('queries');
        $this->fragments = $this->storedQueries('fragments');
    }

    private function storedQueries(string $arrayName): array
    {
        $store = [];
        $finder = new Finder();
        $templates = $finder->files()->in($this->sources[$arrayName]);
        $extension = str_replace('.', '', $this->extension);
        /** @var SplFileInfo $template */
        foreach ($templates as $template) {
            if ($template->getExtension() === $extension) {
                $fragmentName = str_replace($this->extension, '', $template->getFilename());
                $store[$fragmentName] = $template->getContents();
            }
        }
        return $store;
    }

    /**
     * @throws FileNotFoundException|Exception
     */
    public function warmUp(string $cacheDir, ?string $buildDir = null): array
    {
        $this->parser->setFragments($this->fragments);

        $queries = array_map(function ($query) {
            return $this->parser->parseQuery($query);
        }, $this->queries);

        $queries[self::CACHED_FRAGMENT_KEY] = $this->fragments;

        $this->cache->warmUp($queries);

        return array_keys($queries);
    }

    public function isOptional(): bool
    {
        return false;
    }
}
