<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class GraphApiBridge
{
    const TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const EXPIRATION_OFFSET = 3600;

    private HttpClientInterface $client;

    private string $env;

    private GraphApiLogger $logger;

    private array $headers = [];

    private GraphCoverage $coverage;

    private GraphClientFixturesMiddleware $fixturesMiddleware;

    private string $apiUri;

    /**
     * @param HttpClientInterface $graphHttpClient
     * @param GraphApiLogger      $logger
     * @param GraphCoverage       $coverage
     * @param string              $env
     * @param string              $apiUri
     */
    public function __construct(
        HttpClientInterface $graphHttpClient,
        GraphApiLogger $logger,
        GraphCoverage $coverage,
        GraphClientFixturesMiddleware $fixturesMiddleware,
        string $env,
        string $apiUri
    ) {
        $this->client = $graphHttpClient;
        $this->logger = $logger;
        $this->coverage = $coverage;
        $this->fixturesMiddleware = $fixturesMiddleware;
        $this->env = $env;
        $this->apiUri = $apiUri;
    }

    /**
     * @param string      $body
     * @param string      $hash
     * @param string|null $token
     *
     * @throws BadResponseException If the Graph API returns at least one error.
     * @throws \Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface
     *
     * @return \AlloCine\GraphClient\CoverageChecker\ArrayWrapper|\stdClass
     */
    public function query(string $body, string $hash)
    {
        $logId = $this->logger->start($body);
        if ('test' === $this->env) {

            $mode = $this->fixturesMiddleware->getMode();
            if ($mode === 'record') {
                $response = $this->client->request('POST', $this->apiUri, [
                    'body' => $body,
                    'headers' => $this->headers,
                ]);
                $this->fixturesMiddleware->setFixture('POST', $this->apiUri, $hash, $response);
            } else {
                $response = $this->fixturesMiddleware->getFixture('POST', $this->apiUri, [
                    'query' => $hash,
                    'headers' => $this->headers,
                ]);
            }
        } else {
            $response = $this->client->request('POST', $this->apiUri, [
                'body' => $body,
            ]);
        }

        $result = $response->getContent();

        $this->logger->stop($logId, $result);

        $decodedBody = \json_decode($result);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage((object) $decodedBody->data);
    }

    /**
     * @param array $errors
     *
     * @return string
     */
    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }

    /**
     * @param array $headers
     */
    public function addHeaders(array $headers)
    {
        $this->headers = array_merge($this->headers, $headers);
    }
}
