<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class GraphClient
{
    /**
     * @var GraphApiBridge
     */
    private $bridge;

    /**
     * @var GraphCursor
     */
    private $cursor;

    /**
     * @var QueryParser
     */
    private $parser;

    /**
     * @var PhpArrayAdapter
     */
    private $warmUp;

    /**
     * @var string
     */
    private $query;

    /**
     * @var string|null
     */
    private $preparedQuery = null;

    /**
     * @var array
     */
    private $variables = [];

    /**
     * @var string
     */
    private $hash;

    /**
     * GraphClient constructor.
     *
     * @param GraphApiBridge  $bridge
     * @param GraphCursor     $cursor
     * @param QueryParser     $parser
     * @param PhpArrayAdapter $warmUp
     */
    public function __construct(
        GraphApiBridge $bridge,
        GraphCursor $cursor,
        QueryParser $parser,
        PhpArrayAdapter $warmUp
    ) {
        $this->bridge = $bridge;
        $this->cursor = $cursor;
        $this->parser = $parser;
        $this->warmUp = $warmUp;
    }

    /**
     * @param string $query
     */
    private function parseQuery(string $query): void
    {
        if (!empty($this->variables)) {
            $tmpVariables = array_map(function ($key, $value) {
                return sprintf('"%s": %s', $key, $value);
            }, array_keys($this->variables), array_values($this->variables));
            $variables = implode(', ', $tmpVariables);

            if (!empty($variables)) {
                $variables = sprintf(
                    ', "variables": {%s}',
                    $variables
                );
            }
        } else {
            $variables = '';
        }


        $this->query = sprintf(
            '{"query": "%s"%s}',
            addslashes(preg_replace('@\s+@', ' ', $query)),
            $variables
        );
    }

    private function init(): void
    {
        $this->query = null;
        $this->preparedQuery = null;
        $this->hash = null;
        $this->variables = [];
    }

    /**
     * @return string
     */
    public function getQuery(): string
    {
        if (is_null($this->query)) {
            return $this->preparedQuery;
        }

        return $this->query;
    }

    /**
     * @param string $hash
     *
     * @return self
     */
    public function setHash(string $hash): GraphClient
    {
        $this->hash = $hash;

        return $this;
    }

    /**
     * @param array $variables
     *
     * @return self
     */
    public function setVariables(array $variables): GraphClient
    {
        $this->variables = $variables;

        return $this;
    }

    /**
     * @param int $cursor
     * @param int $offset
     *
     * @return null|string
     */
    public function getAfterCursor(int $cursor, int $offset = 0)
    {
        return $this->cursor->getAfterCursor($cursor, $offset);
    }

    /**
     * @param string $query
     *
     * @return GraphClient
     * @throws \FileNotFoundException|\Exception
     */
    public function prepare(string $query): GraphClient
    {
        $this->preparedQuery = $this->parser
            ->setFragments(
                $this->warmUp
                    ->getItem(GraphApiCacheWarmer::CACHED_FRAGMENT_KEY)
                    ->get()
            )
            ->parseQuery($query);

        return $this;
    }

    /**
     * @param string|null $queryName
     *
     * @return self
     * @throws \FileNotFoundException|\Exception
     */
    public function generateQuery(string $queryName = null): GraphClient
    {
        $query = $queryName ? $this->warmUp->getItem($queryName)->get() : null;
        if (is_null($this->preparedQuery) && is_null($query)) {
            throw new \FileNotFoundException("The graph query file $queryName does not exist");
        } elseif (!is_null($this->preparedQuery)) {
            $query = $this->preparedQuery;
        }
        $this->parseQuery($query);

        return $this;
    }

    /**
     * @return \stdClass
     * @throws BadResponseException
     */
    public function getResults(): \stdClass
    {
        $result = $this->bridge->query($this->query, $this->hash);
        $this->init();

        return $result;
    }
}
