<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\FixtureFileNotFoundException;
use AlloCine\GraphClient\Bundle\Sanitizer\RedisKeySanitizer;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Contracts\Cache\CacheInterface;
use Symfony\Contracts\HttpClient\ResponseInterface;

class GraphClientFixturesMiddleware
{
    public function __construct(private readonly CacheInterface $cache,string $apiHost, string $fixturesPath, private readonly string $mode)
    {
    }

    public function getMode(): string
    {
        return $this->mode;
    }

    public function getFixture(string $method, string $url, ?array $options): Response
    {
        $key = $this->getKey($method, $url, $options['query']);
        $fixtureItem = $this->cache->getItem($key);

        if (!$fixtureItem->isHit()) {
            throw new FixtureFileNotFoundException(sprintf('missing file : %s', $key), Response::HTTP_NOT_FOUND);
        }

        return new Response($fixtureItem->get());
    }

    private function getKey(string $method, string $url, string $hash): string
    {
        return RedisKeySanitizer::sanitize(sprintf('%s_%s_%s', $method, $url, $hash));
    }

    public function setFixture(string $method, string $url, string $hash, ResponseInterface $response): bool
    {
        $fixtureItem = $this->cache->getItem($this->getKey($method, $url, $hash));
        $fixtureItem->set(($response->getContent()));

        return $this->cache->save($fixtureItem);
    }
}
