<?php

namespace Tests\AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\FileNotFoundException;
use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Client\GraphApiBridge;
use AlloCine\GraphClient\Bundle\Client\GraphClient;
use AlloCine\GraphClient\Bundle\Client\GraphCursor;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class GraphClientTest extends TestCase
{
    /**
     * @var GraphApiBridge
     */
    private $bridge;

    /**
     * @var GraphCursor
     */
    private $cursor;

    /**
     * @var PhpArrayAdapter
     */
    private $cacheAdapter;

    /**
     * @var FilesystemAdapter
     */
    private $fileSystemAdapter;

    /**
     * @throws \Exception|FileNotFoundException
     */
    public function setUp(): void
    {
        $this->setCacheFragments();

        $this->bridge = $this->getMockBuilder(GraphApiBridge::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->cursor = new GraphCursor();
    }

    public function testCursorValues()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $this->assertNull($service->getAfterCursor(1, 0));
        $this->assertSame(base64_encode('arrayconnection:9'), $service->getAfterCursor(10));
        $this->assertSame(base64_encode('arrayconnection:29'), $service->getAfterCursor(30));
        $this->assertSame(base64_encode('arrayconnection:19'), $service->getAfterCursor(20));
        $this->assertSame(base64_encode('arrayconnection:22'), $service->getAfterCursor(20, 3));
    }

    public function testPrepareError()
    {
        $this->expectException(FileNotFoundException::class);
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testWithError {
                ...errorFragment
            }
        }
GRAPHQL;
        $service->prepare($query);
        $this->assertSame($query, $service->getQuery());
    }

    /**
     * @throws \Exception|\AlloCine\GraphClient\Bundle\Exception\FileNotFoundException
     */
    public function testPrepareWithoutFragment()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query);
        $this->assertSame($query, $service->getQuery());
    }

    /**
     * @throws \Exception|\AlloCine\GraphClient\Bundle\Exception\FileNotFoundException
     */
    public function testPrepareWithFragment()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                ...realFragment
            }
        }
GRAPHQL;
        $expected =
            <<<GRAPHQL
        {
            testReal {
                ...realFragment
            }
        } fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL;
        $service->prepare($query);
        $this->assertSame(rtrim($expected), rtrim($service->getQuery()));
    }

    /**
     * @throws \Exception|\AlloCine\GraphClient\Bundle\Exception\FileNotFoundException
     */
    public function testGenerateQuery()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query)->generateQuery();
        $this->assertSame('{"query": " { testReal { test } }"}', $service->getQuery());
    }

    /**
     * @throws \Exception|\AlloCine\GraphClient\Bundle\Exception\FileNotFoundException
     */
    public function testGenerateQueryWithVariables()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query)->setVariables(['order' => 'ORDER', 'first' => 4, 'bool' => true, 'facets' => [["field" => "ELEMENTS"]]])->generateQuery();
        $this->assertSame('{"query": " { testReal { test } }", "variables": {"order":"ORDER","first":4,"bool":true,"facets":[{"field":"ELEMENTS"}]}}', $service->getQuery());
    }

    /**
     * @throws \Exception|\AlloCine\GraphClient\Bundle\Exception\FileNotFoundException
     */
    private function setCacheFragments()
    {
        $cacheDir = __DIR__ . '/../Resources/cache/queries.php';
        $this->fileSystemAdapter = new FilesystemAdapter('', 0, __DIR__ . '/../Resources/cache/filesystem');
        $this->cacheAdapter = new PhpArrayAdapter($cacheDir, $this->fileSystemAdapter);
        $sources = ['queries' => [__DIR__ . '/../Resources/graph/queries'], 'fragments' => [__DIR__ . '/../Resources/graph/fragments']];
        $service = new GraphApiCacheWarmer($this->cacheAdapter, new QueryParser(), $sources, '.graphql');
        $service->warmUp('');
    }
}
