<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Parser;

use Exception;
use AlloCine\GraphClient\Bundle\Exception\FileNotFoundException;
use GraphQL\Language\AST\FragmentSpreadNode;
use GraphQL\Language\AST\NodeKind;
use GraphQL\Language\Parser;
use GraphQL\Language\Visitor;

class QueryParser
{
    private array $loadedFragments = [];

    private array $fragments = [];

    /**
     * @throws Exception
     */
    private function getFragmentNames(string $source): array
    {
        $fragments = [];
        Visitor::visit(Parser::parse($source), [
            'leave' => [
                NodeKind::FRAGMENT_SPREAD => function (FragmentSpreadNode $node) use (&$fragments): void {
                    $fragments[] = $node->name->value;
                },
            ],
        ]);

        return $fragments;
    }

    /**
     * @throws FileNotFoundException
     */
    private function recursiveAddFragments(string $query): string
    {
        if ($matches = $this->getFragmentNames($query)) {
            foreach ($matches as $fragment) {
                if (!in_array($fragment, $this->loadedFragments)) {
                    if (!array_key_exists($fragment, $this->fragments)) {
                        throw new FileNotFoundException(sprintf('The graph fragment %s does not exist', $fragment));
                    }

                    $fragmentFile = $this->fragments[$fragment];

                    $this->loadedFragments[] = $fragment;
                    $parsedFragment = $this->recursiveAddFragments($fragmentFile);
                    $query .= ' ' . $parsedFragment;
                }
            }
        }

        return $query;
    }

    public function setFragments(array $fragments): QueryParser
    {
        $this->fragments = $fragments;

        return $this;
    }

    /**
     * @throws FileNotFoundException
     */
    public function parseQuery(string $query): string
    {
        $query = $this->recursiveAddFragments($query);
        $this->loadedFragments = [];

        return $query;
    }
}
