<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use GuzzleHttp\Client;
use Symfony\Component\Cache\Adapter\AdapterInterface as Cache;

class GraphApiBridge
{
    const TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const EXPIRATION_OFFSET = 3600;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $token;

    /**
     * @var string
     */
    private $apiHost;

    /**
     * @var string
     */
    private $apiUri;

    /**
     * @var array
     */
    private $headers;

    /**
     * @var string
     */
    private $env;

    /**
     * @var Cache
     */
    private $cache;

    /**
     * @var GraphApiLogger
     */
    private $logger;

    /**
     * @var GraphCoverage
     */
    private $coverage;

    /**
     * @param Client         $client
     * @param Cache          $cache
     * @param GraphApiLogger $logger
     * @param GraphCoverage  $coverage
     * @param string         $env
     * @param string         $token
     * @param string         $apiHost
     * @param string         $apiUri
     * @param array          $headers
     */
    public function __construct(
        Client $client,
        Cache $cache,
        GraphApiLogger $logger,
        GraphCoverage $coverage,
        string $env,
        string $token,
        string $apiHost,
        string $apiUri,
        array $headers = []
    ) {
        $this->client = $client;
        $this->cache = $cache;
        $this->logger = $logger;
        $this->coverage = $coverage;
        $this->env = $env;
        $this->token = $token;
        $this->apiHost = $apiHost;
        $this->apiUri = $apiUri;
        $this->headers = $headers;
    }

    /**
     * @param string $body
     * @param string $hash
     *
     * @return \AlloCine\GraphClient\CoverageChecker\ArrayWrapper|\stdClass
     * @throws BadResponseException|\GuzzleHttp\Exception\GuzzleException
     */
    public function query(string $body, string $hash)
    {
        $headers = $this->getHeaders($this->token);

        $logId = $this->logger->start($body);
        if ($this->env === 'test') {
            $response = $this->client->post($this->apiUri, [
                'body' => $body,
                'query' => $hash,
                'headers' => $headers
            ]);
        } else {
            $client = new Client(['base_uri' => $this->apiHost]);
            $response = $client->request('POST', $this->apiUri, [
                'body' => $body,
                'headers' => $headers
            ]);
        }

        $result = $response ->getBody()->getContents();
        $this->logger->stop($logId, $result);

        $decodedBody = json_decode($result);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage($decodedBody->data);
    }

    /**
     * @param string $token
     *
     * @return array
     */
    private function getHeaders(string $token): array
    {
        return array_merge(
            [
                'content-type'  => 'application/json',
                'authorization' => sprintf('Bearer %s', $token),
            ],
            $this->headers
        );
    }

    /**
     * @param array $errors
     *
     * @return string
     */
    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }
}
