<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use GuzzleHttp\Client;
use Symfony\Component\Cache\Adapter\AdapterInterface as Cache;

class GraphApiBridge
{
    const TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const EXPIRATION_OFFSET = 3600;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $token;

    /**
     * @var string
     */
    private $apiUri;

    /**
     * @var Cache
     */
    private $cache;

    /**
     * @var GraphApiLogger
     */
    private $logger;

    /**
     * @var GraphCoverage
     */
    private $coverage;

    /**
     * @param Client              $client
     * @param Cache               $cache
     * @param GraphApiLogger      $logger
     * @param GraphCoverage       $coverage
     * @param string              $token
     * @param string              $apiUri
     */
    public function __construct(
        Client $client,
        Cache $cache,
        GraphApiLogger $logger,
        GraphCoverage $coverage,
        string $token,
        string $apiUri
    ) {
        $this->client = $client;
        $this->cache = $cache;
        $this->logger = $logger;
        $this->coverage = $coverage;
        $this->token = $token;
        $this->apiUri = $apiUri;
    }

    /**
     * @param string $body
     * @param string $hash
     *
     * @throws BadResponseException If the Graph API returns at least one error.
     *
     * @return Object
     */
    public function query(string $body, string $hash)
    {
        $body = $this->normalizeQuery($body);

        $headers = $this->getHeaders($this->token);

        $logId = $this->logger->start($body);
        $result = $this->client->post($this->apiUri, [
                'body' => $body,
                'query' => $hash,
                'headers' => $headers
            ])
            ->getBody()
            ->getContents();
        $this->logger->stop($logId, $result);

        $decodedBody = json_decode($result);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage($decodedBody->data);
    }

    /**
     * @param string $token
     *
     * @return array
     */
    private function getHeaders(string $token): array
    {
        return [
            'content-type'  => 'application/json',
            'authorization' => sprintf('Bearer %s', $token),
        ];
    }

    /**
     * @param array $errors
     *
     * @return string
     */
    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }

    /**
     * @param string $body
     *
     * @return string
     */
    private function normalizeQuery(string $body): string
    {
        return sprintf('{"query": "%s"}', addslashes(preg_replace('!\s+!', ' ', $body)));
    }
}
