<?php

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Response;
use Symfony\Component\Cache\Adapter\AdapterInterface as Cache;

class GraphApiBridge
{
    const TOKEN_CACHE_KEY = 'token';

    /**
     * This determines how early the token is forcefully regenerated before it's real expiration date.
     * This is to avoid a token that would expire between this class' check and its real usage on the API.
     */
    const EXPIRATION_OFFSET = 3600;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $token;

    /**
     * @var string
     */
    private $apiHost;

    /**
     * @var string
     */
    private $apiUri;

    /**
     * @var string
     */
    private $env;

    /**
     * @var Cache
     */
    private $cache;

    /**
     * @var GraphApiLogger
     */
    private $logger;

    /**
     * @var GraphCoverage
     */
    private $coverage;

    /**
     * @param Client         $client
     * @param Cache          $cache
     * @param GraphApiLogger $logger
     * @param GraphCoverage  $coverage
     * @param string         $token
     * @param string         $apiHost
     * @param string         $apiUri
     * @param string         $env
     */
    public function __construct(
        Client $client,
        Cache $cache,
        GraphApiLogger $logger,
        GraphCoverage $coverage,
        string $env,
        string $token,
        string $apiHost,
        string $apiUri
    ) {
        $this->client = $client;
        $this->cache = $cache;
        $this->logger = $logger;
        $this->coverage = $coverage;
        $this->env = $env;
        $this->token = $token;
        $this->apiHost = $apiHost;
        $this->apiUri = $apiUri;
    }

    /**
     * @param string $body
     * @param string $hash
     *
     * @throws BadResponseException If the Graph API returns at least one error.
     *
     * @return Object
     */
    public function query(string $body, string $hash)
    {
        $logId = $this->logger->start($body);
        $body = $this->normalizeQuery($body);
        if ($this->env === 'test') {
            $headers = $this->getHeaders();
            $response = $this->client->post($this->apiUri, [
                'body'    => $body,
                'query'   => $hash,
                'headers' => $headers,
            ]);
        } else {
            $response = $this->request($body);
        }

        $result = $response->getBody()->getContents();
        $this->logger->stop($logId, $result);

        $decodedBody = json_decode($result);

        if (isset($decodedBody->errors)) {
            throw new BadResponseException($this->parseResponseErrors($decodedBody->errors));
        }

        return $this->coverage->transformToCoverage($decodedBody->data);
    }

    /**
     * @param boolean $withKeys
     *
     * @return array
     */
    private function getHeaders(bool $withKeys = true): array
    {
        if ($withKeys) {
            return [
                'content-type'  => 'application/json',
                'authorization' => sprintf('Bearer %s', $this->token),
            ];
        } else {
            return [
                'content-type: application/json',
                sprintf('authorization: Bearer %s', $this->token),
            ];
        }
    }

    /**
     * @param array $errors
     *
     * @return string
     */
    private function parseResponseErrors(array $errors): string
    {
        $errorMessage = 'Wrong call to graphAPI cause:';

        foreach ($errors as $error) {
            $errorMessage .= ' ==> ' . $error->message . '\n';
        }

        return $errorMessage;
    }

    /**
     * @param string $body
     *
     * @return string
     */
    private function normalizeQuery(string $body): string
    {
        return sprintf('{"query": "%s"}', addslashes(preg_replace('!\s+!', ' ', $body)));
    }

    /**
     * @param $body
     *
     * @return Response
     * @throws BadResponseException
     */
    private function request($body): Response
    {
        $request = curl_init();
        curl_setopt($request, CURLOPT_POST, true);
        curl_setopt($request, CURLOPT_POSTFIELDS, $body);
        curl_setopt_array($request, [
            CURLOPT_URL            => $this->encodeUrl(),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS      => 10,
            CURLOPT_HTTPHEADER     => $this->getHeaders(false),
            CURLOPT_HEADER         => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_ENCODING       => '',
        ]);

        $response = curl_exec($request);
        $error = curl_error($request);
        $info = curl_getinfo($request);
        curl_close($request);

        if ($error) {
            throw new BadResponseException($error);
        }

        $header_size = $info['header_size'];
        $header = substr($response, 0, $header_size);
        $body = substr($response, $header_size);
        $httpCode = $info['http_code'];

        return new Response($httpCode, $this->parseHeaders($header), $body);
    }

    /**
     * @return string
     */
    private function encodeUrl(): string
    {
        $lastCharHost = substr($this->apiHost, -1);
        $firstCharUri = substr($this->apiUri, 0, 1);

        if ($lastCharHost === '/' && $lastCharHost === $firstCharUri) {
            return substr($this->apiHost, 0, -1) . $this->apiUri;
        } elseif ($lastCharHost !== '/' && $firstCharUri !== '/') {
            return $this->apiHost . '/' . $this->apiUri;
        }

        return $this->apiHost . $this->apiUri;
    }

    /**
     * @param string $rawHeaders
     *
     * @return array
     */
    private function parseHeaders(string $rawHeaders): array
    {
        if (function_exists('http_parse_headers')) {
            return http_parse_headers($rawHeaders);
        } else {
            $key = '';
            $headers = [];
            foreach (explode("\n", $rawHeaders) as $i => $h) {
                $h = explode(':', $h, 2);
                if (isset($h[1])) {
                    if (!isset($headers[$h[0]])) {
                        $headers[$h[0]] = trim($h[1]);
                    } elseif (is_array($headers[$h[0]])) {
                        $headers[$h[0]] = array_merge($headers[$h[0]], [trim($h[1])]);
                    } else {
                        $headers[$h[0]] = array_merge([$headers[$h[0]]], [trim($h[1])]);
                    }
                    $key = $h[0];
                } else {
                    if (substr($h[0], 0, 1) == "\t") {
                        $headers[$key] .= "\r\n\t" . trim($h[0]);
                    } elseif (!$key) {
                        $headers[0] = trim($h[0]);
                    }
                }
            }

            return $headers;
        }
    }
}
