<?php

namespace AlloCine\GraphClient\CoverageChecker;

class ArrayWrapper implements \ArrayAccess
{
    /**
     * @var array
     */
    private $source;

    /**
     * @var array
     */
    private $usage;

    /**
     * @var string
     */
    private $path = '';

    /**
     * @param array|\stdClass $source
     * @param string          $path
     */
    public function __construct($source, $path = '')
    {
        $this->path = $path;

        if ($source instanceof \stdClass) {
            $source = $this->arrayCastRecursive($source);
        }

        $this->usage = array_map(function ($item) {
            return new Usage($item);
        }, $source);

        $this->source = [];

        foreach ($source as $name => $value) {
            $subPath = $path ? sprintf('%s.%s', $path, $name) : $name;
            $this->source[$name] = is_array($value) ? new ArrayWrapper($value, $subPath) : $value;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function offsetExists($offset)
    {
        return array_key_exists($offset, $this->source);
    }

    /**
     * {@inheritdoc}
     */
    public function __isset($name)
    {
        return $this->offsetExists($name);
    }

    /**
     * {@inheritdoc}
     */
    public function offsetGet($offset)
    {
        if (!$this->offsetExists($offset)) {
            throw new \OutOfBoundsException(sprintf('Undefined offset "%s"', $offset));
        }

        $this->usage[$offset]->markUsed();

        return $this->source[$offset] ?? null;
    }

    /**
     * {@inheritdoc}
     */
    public function __get(string $name)
    {
        return $this->offsetGet($name);
    }

    /**
     * {@inheritdoc}
     */
    public function offsetSet($offset, $value)
    {
        if (is_array($value)) {
            $value = new ArrayWrapper($value);
        }

        $this->source[$offset] = $value;
    }

    /**
     * {@inheritdoc}
     */
    public function __set($name, $value)
    {
        $this->offsetSet($name, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function offsetUnset($offset)
    {
        if ($this->source[$offset] ?? false) {
            unset($this->source[$offset]);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function __unset($name)
    {
        $this->offsetUnset($name);
    }

    /**
     * Iterates the source array without triggering the usage detection.
     *
     * @param callable $callback
     */
    public function silkTouchIterate(callable $callback)
    {
        foreach ($this->source as $key => $item) {
            call_user_func($callback, $item);
        }
    }

    /**
     * @return string
     */
    public function getPath(): string
    {
        return $this->path;
    }

    /**
     * @return array
     */
    public function getCoverage(): array
    {
        $uncovered = 0;
        $covered = 0;

        foreach ($this->usage as $key => $used) {
            if ($this->source[$key] instanceof ArrayWrapper) {
                $coverage = $this->source[$key]->getCoverage();
                $covered += $coverage['covered'];
                $uncovered += $coverage['uncovered'];
            } else {
                if ($used->isUsed()) {
                    $covered += $used->getWeight();
                } else {
                    $uncovered += $used->getWeight();
                }
            }
        }

        return [
            'covered'   => $covered,
            'uncovered' => $uncovered,
        ];
    }

    /**
     * @param $array
     *
     * @return array
     */
    private function arrayCastRecursive($array): array
    {
        if (is_array($array)) {
            foreach ($array as $key => $value) {
                if (is_array($value)) {
                    $array[$key] = $this->arrayCastRecursive($value);
                }
                if ($value instanceof \stdClass) {
                    $array[$key] = $this->arrayCastRecursive((array)$value);
                }
            }
        }
        if ($array instanceof \stdClass) {
            return $this->arrayCastRecursive((array)$array);
        }

        return $array;
    }
}
